import bpy
import numpy as np
import bmesh
from mathutils import kdtree, bvhtree, Vector, Matrix
from bpy.props import *
from bpy.types import Operator
from bpy.utils import register_classes_factory
from .properties import *


#
# FUNCTIONS
#


def refresh_bm(bm):
    bm.edges.ensure_lookup_table()
    bm.verts.ensure_lookup_table()
    bm.faces.ensure_lookup_table()
    return


def ob_to_bm_world(ob, apply_mods, world, tri):
    deps_g = bpy.context.evaluated_depsgraph_get()

    # Convert ob to bm world space
    bm = bmesh.new()
    if apply_mods:
        ob_ev = ob.evaluated_get(deps_g)
        conv = ob_ev.to_mesh()
        bm.from_mesh(conv)

        ob_ev.to_mesh_clear()
    else:
        bm.from_mesh(ob.data)

    if tri:
        bmesh.ops.triangulate(bm, faces=bm.faces)

    refresh_bm(bm)

    # bm in world space
    if world:
        bm.transform(ob.matrix_world)

    #bmesh.ops.triangulate(bm, faces=bm.faces)
    bm.normal_update()

    return bm


def create_kd_from_bm(mesh):
    # KD Tree
    size = len(mesh.verts)
    kd = kdtree.KDTree(size)
    for i, v in enumerate(mesh.verts):
        kd.insert(v.co, i)
    kd.balance()

    return kd


def create_kd(mesh):
    # KD Tree
    size = len(mesh.vertices)
    kd = kdtree.KDTree(size)
    for i, v in enumerate(mesh.vertices):
        kd.insert(v.co, i)
    kd.balance()

    return kd


def get_edge_angle_order(vec, edges, indeces, start_co, angle_limit=180.0):
    order = []
    values = []

    if vec.length < .00001:
        pass

    for e, ed_cos in enumerate(edges):
        if (ed_cos[0]-start_co).length < .001:
            ed_vec = ed_cos[1]-ed_cos[0]
        else:
            ed_vec = ed_cos[0]-ed_cos[1]

        # Edge is zero length
        if ed_vec.length < .00001:

            # Test edge is also zero length so give it a match
            if vec.length < .00001:
                ed_ang = 0.0
            # Test edge has some length so give it lowest priority
            else:
                ed_ang = 180.0

        # Edge has some length
        else:

            # Test edge is zero lnegth so give it lowest priority
            if vec.length < .00001:
                ed_ang = 180.0
            # Test edge has length so test the angle as normal
            else:
                ed_ang = vec.angle(ed_vec)

        if ed_ang <= angle_limit:
            order.append(indeces[e])
            values.append(ed_ang)

    sorted_order = [ind for val, ind in sorted(zip(values, order))]
    return sorted_order


#


def create_new_shape_key(key, ob, mirror_key, mirror, duplicate):
    scn_prop = bpy.context.scene.skmp_props

    # Add string for mirror or duplicate
    if mirror:
        new_name = key.name + '_MIRROR'
    else:
        new_name = key.name + '_DUPLI'

    # Create new name if renaming
    if scn_prop.text_rename != '' and scn_prop.text_filter != '' and scn_prop.text_filter in key.name:
        new_name = key.name.replace(scn_prop.text_filter, scn_prop.text_rename)

    # Duplicate if decessary
    if duplicate:
        new_key = ob.shape_key_add(name=new_name, from_mix=False)
    else:
        new_key = key

    # Set min max on new key
    new_key.slider_min = key.slider_min
    new_key.slider_max = key.slider_max

    # Create new mirror/dup shape key displacement data

    if mirror:
        new_data = [v.co.copy() for v in ob.data.vertices]
        for i, data in enumerate(key.data):

            # if key_vec.length > .000001:
            if scn_prop.absoulte_mirror_coordinates:
                new_co = key.data[mirror_key[i]].co.copy()

                new_co[scn_prop.mirror_axis] = new_co[scn_prop.mirror_axis] * -1
                new_data[i] = new_co

            else:
                key_vec = key.data[mirror_key[i]].co - \
                    ob.data.vertices[mirror_key[i]].co

                key_vec[scn_prop.mirror_axis] = key_vec[scn_prop.mirror_axis] * -1
                new_data[i] = ob.data.vertices[i].co + key_vec
    else:
        new_data = [v.co.copy() for v in key.data]

    for i, co in enumerate(new_data):
        if scn_prop.selected_only:
            if ob.data.vertices[i].select == False:
                continue
        new_key.data[i].co = co

    if duplicate and scn_prop.copy_drivers:
        if ob.data.shape_keys.animation_data != None:
            copy_drivers(ob, ob, key, new_key)

    if scn_prop.vgroup != '':
        new_key.vertex_group = scn_prop.vgroup

    return


def get_mirror_indeces(ob, kdtree, axis, force_mirror=False):
    mirror_indeces = []
    for v in ob.data.vertices:
        mirror_co = v.co.copy()
        mirror_co[axis] = mirror_co[axis] * -1

        # Only get first vert from mirror co if forcing
        if force_mirror:
            kd_res = kdtree.find(mirror_co)
            mirror_indeces.append(kd_res[1])
            found = True
        else:
            kd_res = kdtree.find_n(mirror_co, 25)
            found = False
            for (co, index, dist) in kd_res:
                if index not in mirror_indeces:
                    found = True
                    mirror_indeces.append(index)
                break

        if found == False:
            return False

    return mirror_indeces


def get_topology_mirror_indeces(ob, kdtree, axis):
    #
    # Match 2 objects with same topology but slightly altered vertex coordinates
    #
    ob_bm = ob_to_bm_world(ob, False, False, False)

    mirror_indeces = []
    # Find verts with a threshold distance that have same # of loops
    for v in ob_bm.verts:
        mirror_co = v.co.copy()
        mirror_co[axis] *= -1

        co, index, dist = kdtree.find(mirror_co)
        if dist < .1:
            if len(ob_bm.verts[index].link_loops) == len(v.link_loops):
                mirror_indeces.append(index)
            else:
                mirror_indeces.append(None)
        else:
            mirror_indeces.append(None)

    searching = True
    while searching:
        new_link = 0
        missing_links = []

        # Get a list of verts on active bmesh that have a match to current bm and are connected to unmatched verts
        for v in ob_bm.verts:
            if mirror_indeces[v.index] != None:
                for e, ed in enumerate(v.link_edges):
                    ov = ed.other_vert(v)
                    if mirror_indeces[ov.index] == None and v.index not in missing_links:
                        missing_links.append(v.index)

        # create matches between the missing data using the matchverts their linked edges comparing similar edge direction
        for ind in missing_links:
            v = ob_bm.verts[ind]
            lv = ob_bm.verts[mirror_indeces[v.index]]

            avail_eds = [
                ed for ed in v.link_edges if mirror_indeces[ed.other_vert(v).index] == None]
            avail_leds = [
                ed for ed in lv.link_edges if mirror_indeces[ed.other_vert(lv).index] == None]
            l_vecs = [[ed.verts[0].co, ed.verts[1].co] for ed in avail_leds]
            l_inds = [ed.index for ed in avail_leds]

            if len(l_inds) > 0 and len(avail_eds) > 0:
                for ed in avail_eds:
                    ov = ed.other_vert(v)
                    vec = ov.co - v.co
                    vec[axis] *= -1

                    ang_order = get_edge_angle_order(
                        vec, l_vecs, l_inds, v.co)

                    match_ed = ob_bm.edges[ang_order[0]]
                    lov = match_ed.other_vert(lv)

                    mirror_indeces[ov.index] = lov.index
                    mirror_indeces[lov.index] = ov.index
                    new_link += 1

        if new_link == 0:
            searching = False

    if None in mirror_indeces:
        return False

    return mirror_indeces


def np_barycentric_weights(tri_v1, tri_v2, tri_v3, point):
    #
    # Get the barycentric weights of a point inside a triangle
    # This works for points outside of triangle as well
    # Code based on this page https://www.cdsimpson.net/2014/10/barycentric-coordinates.html
    #

    # Original only works on inside
    # ed1 = tri_v2 - tri_v1
    # ed2 = tri_v3 - tri_v1
    # area = ed1.cross(ed2).length / 2

    # vec1 = tri_v1 - point
    # vec2 = tri_v2 - point
    # vec3 = tri_v3 - point

    # u = (vec2.cross(vec3).length / 2) / area
    # v = (vec1.cross(vec3).length / 2) / area
    # w = 1 - v - u

    weights = np.empty(tri_v1.shape[0]*3, dtype=np.float32).reshape(-1, 3)

    vap = point - tri_v1
    vbp = point - tri_v2
    vcp = point - tri_v3

    vac = tri_v3 - tri_v1
    vab = tri_v2 - tri_v1
    vca = tri_v1 - tri_v3
    vbc = tri_v3 - tri_v2

    n = np.cross(vab, vac)
    na = np.cross(vbc, vbp)
    nb = np.cross(vca, vcp)
    nc = np.cross(vab, vap)

    n_dot = np.sum(np.square(n), axis=1)

    weights[:, 0] = (np.sum(n * na, axis=1)) / n_dot
    weights[:, 1] = (np.sum(n * nb, axis=1)) / n_dot
    weights[:, 2] = (np.sum(n * nc, axis=1)) / n_dot
    return weights


def match_data(base, targ):
    scn_prop = bpy.context.scene.skmp_props

    bvh = bvhtree.BVHTree.FromBMesh(targ)

    selection = []
    linked = []
    # loop verts
    for vert in base.verts:
        if vert.select == False and scn_prop.limit_selected:
            continue

        # nearest vert in neutral data
        co, norm, ind, dist = bvh.find_nearest(vert.co)

        weights = np_barycentric_weights(
            np.array(targ.faces[ind].verts[0].co).reshape(-1, 3),
            np.array(targ.faces[ind].verts[1].co).reshape(-1, 3),
            np.array(targ.faces[ind].verts[2].co).reshape(-1, 3),
            np.array(co).reshape(-1, 3))

        selection.append(vert.index)
        linked.append([
            [targ.faces[ind].verts[0].index,
             targ.faces[ind].verts[1].index,
             targ.faces[ind].verts[2].index],
            dist,
            weights[0]])

    return linked, selection


def compare_topology(active_ob, ob):
    #
    # Match 2 objects with same topology but slightly altered vertex coordinates
    #
    act_bm = ob_to_bm_world(active_ob, False, True, False)

    ob_bm = ob_to_bm_world(ob, False, True, False)
    kd = create_kd(ob.data)

    matching = []
    # Find verts with a threshold distance that have same # of loops
    for v in act_bm.verts:
        co, index, dist = kd.find(v.co)
        if dist < .1:
            if len(ob_bm.verts[index].link_loops) == len(v.link_loops):
                matching.append(index)
            else:
                matching.append(None)
        else:
            matching.append(None)

    searching = True
    while searching:
        new_link = 0
        missing_links = []

        # Get a list of verts on active bmesh that have a match to current bm and are connected to unmatched verts
        for v in act_bm.verts:
            if matching[v.index] != None:
                for e, ed in enumerate(v.link_edges):
                    ov = ed.other_vert(v)
                    if matching[ov.index] == None and v.index not in missing_links:
                        missing_links.append(v.index)

        # create matches between the missing data using the matchverts their linked edges order
        for ind in missing_links:
            v = act_bm.verts[ind]
            lv = ob_bm.verts[matching[v.index]]
            connect = None
            other_connect = None

            for i, ed in enumerate(v.link_edges):
                ov = ed.other_vert(v)

                if matching[ov.index] == None:
                    connect = ov.index
                    break

            for i, ed in enumerate(lv.link_edges):
                ov = ed.other_vert(lv)

                if ov.index not in matching:
                    other_connect = ov.index
                    break

            if connect != None and other_connect != None:
                matching[connect] = other_connect

                new_link += 1

        if new_link == 0:
            searching = False

    return matching

#


def driver_settings_copy(copy_drv, tar_drv):
    scn_prop = bpy.context.scene.skmp_props

    tar_drv.driver.type = copy_drv.driver.type
    tar_drv.driver.use_self = copy_drv.driver.use_self

    for var in copy_drv.driver.variables:
        new_var = tar_drv.driver.variables.new()
        new_var.name = var.name
        new_var.type = var.type

        count = 0
        for tar in var.targets:
            new_var.targets[count].bone_target = tar.bone_target
            new_var.targets[count].data_path = tar.data_path

            if scn_prop.rename_driver_bones:
                new_var.targets[count].bone_target = tar.bone_target.replace(
                    scn_prop.text_filter, scn_prop.text_rename)
                new_var.targets[count].data_path = tar.data_path.replace(
                    scn_prop.text_filter, scn_prop.text_rename)

            new_var.targets[count].id = tar.id
            new_var.targets[count].transform_space = tar.transform_space
            new_var.targets[count].transform_type = tar.transform_type

            count += 1

    tar_drv.driver.expression = copy_drv.driver.expression
    return


def copy_drivers(copy_ob, tar_ob, copy_key, tar_key):
    copy_sk = copy_ob.data.shape_keys
    copy_drivers = copy_sk.animation_data.drivers

    if tar_ob.data.animation_data == None:
        tar_ob.data.animation_data_create()
    tar_sk = tar_ob.data.shape_keys

    for drv in copy_drivers:
        drv_name = drv.data_path.replace(
            'key_blocks["', '').replace('"].value', '')

        if copy_key.name == drv_name:
            for sk in tar_sk.key_blocks:
                if sk.name == tar_key.name:
                    new_driver = sk.driver_add('value', -1)
                    driver_settings_copy(drv, new_driver)


#
# OPERATORS
#


class SKMP_OT_rename_filtered_sk(Operator):
    bl_idname = "skmp.rename_filter_sk"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        if aobj.data.shape_keys == None:
            self.report(type={'ERROR'},
                        message='Active Object has no Shape Keys')
            return {'CANCELLED'}

        if scn_prop.text_filter == '':
            self.report(type={'ERROR'}, message='Text Filter not filled out')
            return {'CANCELLED'}

        for key in aobj.data.shape_keys.key_blocks:
            if scn_prop.text_filter in key.name:
                key.name = key.name.replace(
                    scn_prop.text_filter, scn_prop.text_rename)

        return {'FINISHED'}


class SKMP_OT_duplicate_mir_active_sk(Operator):
    bl_idname = "skmp.dupmirror_active_sk"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    execution: StringProperty()

    def execute(self, context):
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        # Check object has shape keys
        if aobj.data.shape_keys == None:
            self.report(type={'ERROR'},
                        message='Active Object has no Shape Keys')
            return {'CANCELLED'}

        # Match verts on mirror axis and end if can't find a match
        kd = create_kd(aobj.data)

        mirror_key = False
        if 'Mir' in self.execution:
            if scn_prop.mirror_method == 'TOPOLOGY_MIRROR':
                mirror_key = get_topology_mirror_indeces(
                    aobj, kd, scn_prop.mirror_axis)
            else:
                mirror_key = get_mirror_indeces(
                    aobj, kd, scn_prop.mirror_axis, force_mirror=scn_prop.mirror_method == 'FORCE_MIRROR')

        # End if no mirror found and trying to mirror
        if mirror_key == False and 'Mir' in self.execution:
            self.report(type={'ERROR'},
                        message='Mesh cannot be mirrored on this axis')
            return {'CANCELLED'}

        create_new_shape_key(aobj.active_shape_key, aobj, mirror_key,
                             'Mir' in self.execution, 'Dup' in self.execution)
        return {'FINISHED'}


class SKMP_OT_duplicate_mir_filtered_sk(Operator):
    bl_idname = "skmp.dupmirror_filter_sk"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    execution: StringProperty()

    def execute(self, context):
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        # Check object has shape keys
        if aobj.data.shape_keys == None:
            self.report(type={'ERROR'},
                        message='Active Object has no Shape Keys')
            return {'CANCELLED'}

        # Match verts on mirror axis and end if can't find a match
        kd = create_kd(aobj.data)

        mirror_key = False
        if 'Mir' in self.execution:
            if scn_prop.mirror_method == 'TOPOLOGY_MIRROR':
                mirror_key = get_topology_mirror_indeces(
                    aobj, kd, scn_prop.mirror_axis)
            else:
                mirror_key = get_mirror_indeces(
                    aobj, kd, scn_prop.mirror_axis, force_mirror=scn_prop.mirror_method == 'FORCE_MIRROR')

        # End if no mirror found and trying to mirror
        if mirror_key == False and 'Mir' in self.execution:
            self.report(type={'ERROR'},
                        message='Mesh cannot be mirrored on this axis')
            return {'CANCELLED'}

        names = [key.name for key in aobj.data.shape_keys.key_blocks]
        for name in names:
            key = aobj.data.shape_keys.key_blocks[name]
            if scn_prop.text_filter != '' and scn_prop.text_filter not in key.name or key == aobj.data.shape_keys.reference_key:
                continue

            create_new_shape_key(key, aobj, mirror_key,
                                 'Mir' in self.execution, 'Dup' in self.execution)

        return {'FINISHED'}


class SKMP_OT_copy_sel_obs_to_sk(Operator):
    bl_idname = "skmp.copy_selected_obj"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        not_copied = []
        for ob in context.selected_objects:
            new_data = []

            if ob == aobj:
                continue

            if len(ob.data.vertices) == len(aobj.data.vertices):
                for v in ob.data.vertices:
                    new_data.append(v.co)
        #    elif scn_prop.continue_anyway == True and len(ob.data.vertices) > len(aobj.data.vertices):
        #        for v in ob.data.vertices:
        #            new_data.append(v.co)
            else:
                #    ## Attempts to read topology for translating data
                #    if scn_prop.data_search:
                #        index_key = compare_topology(aobj, ob)
                #        for i, key in enumerate(index_key):
                #            if key != None:
                #                new_data.append(ob.data.vertices[key].co)
                #            else:
                #                new_data.append(aobj.data.vertices[i].co)
                #    else:
                not_copied.append(ob.name)
                continue

            if aobj.data.shape_keys == None:
                aobj.shape_key_add(name='Basis', from_mix=False)

            new_key = aobj.shape_key_add(name=ob.name, from_mix=False)

            for i, v in enumerate(new_key.data):
                v.co = new_data[i]

        missed = ' '.join(not_copied)

        if len(not_copied) > 0:
            self.report(type={
                        'ERROR'}, message='These objects were not copied because the data did not match:' + missed)
        return {'FINISHED'}


class SKMP_OT_copy_sks_from_sel_obs(Operator):
    bl_idname = "skmp.copy_selected_sks"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        not_copied = []
        no_keys = []
        for ob in context.selected_objects:
            if ob == aobj:
                continue
            if len(ob.data.vertices) != len(aobj.data.vertices):
                not_copied.append(ob.name)
                continue

            if aobj.data.shape_keys == None:
                aobj.shape_key_add(name='Basis', from_mix=False)

            if ob.data.shape_keys == None:
                no_keys.append(ob.name)
                continue

            for sk in ob.data.shape_keys.key_blocks:
                if sk != ob.data.shape_keys.reference_key:
                    sk_name = sk.name

                    new_key = aobj.shape_key_add(
                        name=sk_name, from_mix=False)

                    for i, v in enumerate(new_key.data):
                        v.co = sk.data[i].co

        data_missed = ' '.join(not_copied)
        keys_missed = ' '.join(no_keys)

        if len(not_copied) > 0:
            self.report(type={
                        'ERROR'}, message='These objects were not copied because the data did not match: ' + data_missed)
        if len(no_keys) > 0:
            self.report(type={
                        'ERROR'}, message='These objects were not copied because they have no shape keys: ' + keys_missed)
        return {'FINISHED'}


class SKMP_OT_copy_sks_from_sel_mismatching(Operator):
    bl_idname = "skmp.copy_sks_from_selected_mismatch"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        gen_mods = ['ARRAY', 'BEVEL', 'BOOLEAN', 'BUILD', 'DECIMATE',
                    'EDGE_SPLIT', 'MASK', 'MIRROR', 'MULTIRES', 'REMESH',
                    'SCREW', 'SKIN', 'SOLIDIFY', 'SUBSURF', 'TRIANGULATE', 'WIREFRAME']

        # # turn off generating mods
        # mod_activ = []
        # for mod in aobj.modifiers:
        #     mod_activ.append(mod.show_viewport)
        #     if mod.type in gen_mods:
        #         mod.show_viewport = False

        # convert posed object to data
        aobj_bm = ob_to_bm_world(aobj, True, scn_prop.copy_world_space, False)

        # # reactivate the mods
        # for i, mod in enumerate(aobj.modifiers):
        #     mod.show_viewport = mod_activ[i]

        # Add basis shape key if missing
        if aobj.data.shape_keys == None:
            aobj.shape_key_add(name='Basis', from_mix=False)

        no_keys = []
        for target in context.selected_objects:
            if target == aobj:
                continue

            # If copying object has no shape keys report it to user
            if target.data.shape_keys == None:
                no_keys.append(target.name)
                continue

            # # turn off generating mods
            # mod_activ = []
            # for mod in target.modifiers:
            #     mod_activ.append(mod.show_viewport)
            #     if mod.type in gen_mods:
            #         mod.show_viewport = False

            # store neutral pose object to data
            targ_bm = ob_to_bm_world(
                target, True, scn_prop.copy_world_space, True)

            # # reactivate the mods
            # for i, mod in enumerate(target.modifiers):
            #     mod.show_viewport = mod_activ[i]

            linked_verts, selection = match_data(aobj_bm, targ_bm)

            new_cos = np.zeros(
                len(aobj.data.vertices)*3, dtype=np.float32)

            og_sk_vals = [sk.value for sk in target.data.shape_keys.key_blocks]
            og_sk_mutes = [sk.mute for sk in target.data.shape_keys.key_blocks]
            # 0 sk values
            for s in range(len(og_sk_vals)):
                target.data.shape_keys.key_blocks[s].value = 0.0
                target.data.shape_keys.key_blocks[s].mute = False

            aobj_mat = aobj.matrix_world.copy()
            aobj_mat.translation = [0, 0, 0]

            # transfer shape key data based on linked data
            for sk in target.data.shape_keys.key_blocks:
                if sk != target.data.shape_keys.reference_key:

                    # get sk pose object to data
                    sk.value = 1.0
                    sk_bm = ob_to_bm_world(
                        target, True, scn_prop.copy_world_space, False)

                    sk_cos = np.array(
                        [v.co for v in sk_bm.verts]).reshape(-1, 3)

                    #
                    #
                    #

                    new_sk = aobj.shape_key_add(name=sk.name, from_mix=False)

                    # blank list for foreach_set
                    aobj.data.vertices.foreach_get("co", new_cos)

                    # get vector deformation for all verts required in this sk
                    for i, ind in enumerate(selection):
                        vert = aobj.data.vertices[ind]
                        b_vert = aobj_bm.verts[ind]

                        v_inds = linked_verts[i][0]
                        dist = linked_verts[i][1]
                        weights = linked_verts[i][2]

                        vec1 = sk_cos[v_inds[1]] - sk_cos[v_inds[0]]
                        vec2 = sk_cos[v_inds[2]] - sk_cos[v_inds[1]]

                        v1 = (sk_cos[v_inds[0]] * weights[0])
                        v2 = (sk_cos[v_inds[1]] * weights[1])
                        v3 = (sk_cos[v_inds[2]] * weights[2])

                        bary_norm = Vector(vec1).cross(
                            vec2).normalized() * dist
                        bary_co = v1 + v2 + v3 + bary_norm

                        # difference between neutral and posed coord
                        vec = aobj_mat.inverted() @ (Vector(bary_co) - b_vert.co)

                        # final coord
                        co = vert.co + (vec * scn_prop.copy_scale)

                        # sets the values for the foreach_set
                        new_cos[vert.index*3] = co[0]
                        new_cos[vert.index*3+1] = co[1]
                        new_cos[vert.index*3+2] = co[2]

                    # set the shape_key co values
                    new_sk.data.foreach_set('co', new_cos)

                    # coorective smooth applied to result of bake
                    if scn_prop.smooth_iterations > 0:
                        # mute and store shape keys and modifiers
                        cur_sk_activ = []
                        cur_mod_activ = []

                        for cur_sk in aobj.data.shape_keys.key_blocks:
                            cur_sk_activ.append(cur_sk.mute)
                            if cur_sk != new_sk:
                                cur_sk.mute = True
                            else:
                                cur_sk.mute = False

                        for cur_mod in aobj.modifiers:
                            cur_mod_activ.append(cur_mod.show_viewport)
                            cur_mod.show_viewport = False

                        # add smooth and set settings
                        temp_mod = aobj.modifiers.new(
                            name='TEMP SMOOTH', type='SMOOTH')
                        temp_mod.factor = 0.5
                        temp_mod.iterations = scn_prop.smooth_iterations

                        new_sk.value = 1.0

                        # smoothed mesh to data and set to the shape_key data
                        smooth_bm = ob_to_bm_world(
                            aobj, True, scn_prop.copy_world_space, False)
                        smoothed_mesh = data.meshes.new('Temp mesh')
                        smooth_bm.to_mesh(smoothed_mesh)
                        smoothed_mesh.vertices.foreach_get('co', new_cos)

                        data.meshes.remove(smoothed_mesh)

                        new_sk.value = 0.0

                        # remove smooth
                        aobj.modifiers.remove(temp_mod)

                        # reset the shape key and modifier visibility
                        for x, cur_sk in enumerate(aobj.data.shape_keys.key_blocks):
                            cur_sk.mute = cur_sk_activ[x]
                        for x, cur_mod in enumerate(aobj.modifiers):
                            cur_mod.show_viewport = cur_mod_activ[x]

                        # delete temporary data
                        smooth_bm.free()

                        # set the shape_key co values
                        new_sk.data.foreach_set('co', new_cos)

                    new_sk.slider_min = sk.slider_min
                    new_sk.slider_max = sk.slider_max
                    new_sk.vertex_group = sk.vertex_group
                    if sk.relative_key.name in aobj.data.shape_keys.key_blocks:
                        new_sk.relative_key = aobj.data.shape_keys.key_blocks[sk.relative_key.name]

                    if scn_prop.copy_drivers:
                        if target.data.shape_keys.animation_data != None:
                            copy_drivers(target, aobj, sk, new_sk)

                    sk_bm.free()
                    sk.value = 0.0

            # Reset sk values
            for s in range(len(og_sk_vals)):
                target.data.shape_keys.key_blocks[s].value = og_sk_vals[s]
                target.data.shape_keys.key_blocks[s].mute = og_sk_mutes[s]

            targ_bm.free()

        aobj_bm.free()

        keys_missed = ' '.join(no_keys)
        if len(no_keys) > 0:
            self.report(type={
                        'ERROR'}, message='These objects were not copied because they have no shape keys: ' + keys_missed)

        return {'FINISHED'}


class SKMP_OT_set_mir_axis(Operator):
    bl_idname = "skmp.set_mirror_axis"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    mirror_axis: IntProperty()

    def execute(self, context):
        context.scene.skmp_props.mirror_axis = self.mirror_axis
        return {'FINISHED'}


class SKMP_OT_assign_vertex_group_filtered(Operator):
    bl_idname = "skmp.assign_vgroup_filtered"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        if scn_prop.text_filter != '':
            for sk in aobj.data.shape_keys.key_blocks:
                if scn_prop.text_filter in sk.name:
                    sk.vertex_group = scn_prop.vgroup

        return {'FINISHED'}


class SKMP_OT_toggle_visibility(Operator):
    bl_idname = "skmp.toggle_vis"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    onoroff: BoolProperty()

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        for sk in aobj.data.shape_keys.key_blocks:
            sk.mute = self.onoroff

        return {'FINISHED'}


class SKMP_OT_toggle_visibility_filter(Operator):
    bl_idname = "skmp.toggle_vis_filter"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    onoroff: IntProperty()

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        if self.onoroff == 0 or self.onoroff == 2:
            toggle = True
        else:
            toggle = False
        for sk in aobj.data.shape_keys.key_blocks:
            if self.onoroff == 0 or self.onoroff == 1:
                if scn_prop.text_filter in sk.name:
                    sk.mute = toggle
            else:
                if scn_prop.text_filter not in sk.name:
                    sk.mute = toggle

        return {'FINISHED'}


class SKMP_OT_explode_sks(Operator):
    bl_idname = "skmp.explode_sks"
    bl_label = "Explode Shape Keys to Separate Object"
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        # add in collection if not already existing
        coll_name = 'SKMP Exploded Shape Keys'
        if coll_name not in data.collections:
            data.collections.new(coll_name)

        obj_coll = data.collections[coll_name]

        # link to scene collection if not already
        if obj_coll.name not in scn.collection.children:
            scn.collection.children.link(obj_coll)

        # cache selected vert indices
        sel_verts = [v.index for v in aobj.data.vertices if v.select == True]

        x_size = scn_prop.sep_distance
        x_loc = aobj.location[0] + x_size

        ref_key = aobj.data.shape_keys.reference_key

        for sk in aobj.data.shape_keys.key_blocks:
            cos = []
            if sk != ref_key:
                for i, dat in enumerate(sk.data):
                    fac = 1.0
                    diff = dat.co - aobj.data.vertices[i].co

                    if sk.vertex_group != '' and scn_prop.explode_apply_vgroup:
                        vg = aobj.vertex_groups[sk.vertex_group]
                        try:
                            fac = vg.weight(i)
                        except:
                            fac = 0.0

                    cos.append(diff*fac + aobj.data.vertices[i].co)

                new_mesh = aobj.data.copy()
                new_ob = data.objects.new(sk.name, new_mesh)
                new_ob.shape_key_clear()

                for v, vert in enumerate(new_ob.data.vertices):
                    vert.co = cos[v]

                obj_coll.objects.link(new_ob)

                new_ob.location[0] = x_loc
                x_loc += x_size

                for ed in new_ob.data.edges:
                    ed.select = False
                for p in new_ob.data.polygons:
                    p.select = False
                for v in new_ob.data.vertices:
                    v.select = False

                # restore vertice selection
                for vert in new_ob.data.vertices:
                    if vert.index in sel_verts:
                        vert.select = True

                new_ob.rotation_mode = aobj.rotation_mode
                new_ob.rotation_euler = aobj.rotation_euler
                new_ob.rotation_quaternion = aobj.rotation_quaternion
                new_ob.rotation_axis_angle = aobj.rotation_axis_angle
                new_ob.scale = aobj.scale

        return {'FINISHED'}


class SKMP_OT_refresh_relations(Operator):
    bl_idname = "skmp.refresh_relations"
    bl_label = "Refresh Shape Key Relationships"
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        for relation in aobj.skmp_par_coll.shape_key_coll:
            if relation.parent_sk != '' and relation.child_sk != '':
                kbs = aobj.data.shape_keys.key_blocks
                if relation.parent_sk in kbs and relation.child_sk in kbs:
                    par = kbs[relation.parent_sk]
                    child = kbs[relation.child_sk]
                    for i, dat in enumerate(par.data):
                        child.data[i].co = dat.co

        return {'FINISHED'}


class SKMP_OT_apply_shape_keys(Operator):
    bl_idname = "skmp.apply_shape_keys"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        app_sk = aobj.shape_key_add(name='Applied SKS', from_mix=True)

        for i in range(len(aobj.data.shape_keys.key_blocks)):
            if aobj.data.shape_keys.key_blocks[0].name != app_sk.name:
                aobj.shape_key_remove(aobj.data.shape_keys.key_blocks[0])

        aobj.shape_key_clear()

        return {'FINISHED'}


class SKMP_OT_set_active_as_basis(Operator):
    bl_idname = "skmp.set_active_as_basis"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        act_sk = aobj.active_shape_key
        ref_sk = aobj.data.shape_keys.reference_key

        act_defs = []
        for d, dat in enumerate(ref_sk.data):
            #act_defs.append( (act_sk.data[d].co - dat.co).copy() )
            act_defs.append(act_sk.data[d].co - dat.co)

        for sk in aobj.data.shape_keys.key_blocks:
            if sk != act_sk and sk != ref_sk:
                sk_defs = []
                for d, dat in enumerate(ref_sk.data):
                    #sk_defs.append( (sk.data[d].co - dat.co).copy() )
                    sk_defs.append(sk.data[d].co - dat.co)

                for d, dat in enumerate(sk.data):
                    dat.co = ref_sk.data[d].co + act_defs[d] + sk_defs[d]

        while aobj.data.shape_keys.reference_key.name != act_sk.name:
            bpy.ops.object.shape_key_move(type='UP')

        for sk in aobj.data.shape_keys.key_blocks:
            if sk != act_sk:
                sk.relative_key = act_sk

        return {'FINISHED'}


# class SKMP_OT_set_active_as_basis(Operator):
#     bl_idname = "skmp.set_active_as_basis"
#     bl_label = ""
#     bl_options = {"REGISTER", "UNDO", "INTERNAL"}

#     def execute(self, context):
#         data = bpy.data
#         context = bpy.context

#         obs = data.objects
#         aobj = context.active_object
#         scn = context.scene
#         scn_prop = scn.skmp_props

#         act_sk = aobj.active_shape_key
#         ref_sk = aobj.data.shape_keys.reference_key

#         # act_defs = []
#         # for d, dat in enumerate(ref_sk.data):
#         #     #act_defs.append( (act_sk.data[d].co - dat.co).copy() )
#         #     # act_defs.append(act_sk.data[d].co - dat.co)
#         #     if (ref_sk.data[d].co - dat.co).length > .0001:
#         #         act_defs.append(dat.co)
#         #     else:
#         #         act_defs.append(None)

#         # ref_defs = []
#         # for d, dat in enumerate(ref_sk.data):
#         #     #act_defs.append( (act_sk.data[d].co - dat.co).copy() )
#         #     # act_defs.append(act_sk.data[d].co - dat.co)
#         #     ref_defs.append(dat.co.copy())

#         act_cos = []
#         for d in act_sk.data:
#             act_cos.append(d.co)

#         def_inds, def_cos, sks = [], [], []
#         for sk in aobj.data.shape_keys.key_blocks:
#             if sk != act_sk and sk != ref_sk:
#                 inds, cos = [], []
#                 for d, dat in enumerate(sk.data):
#                     if (dat.co - ref_sk.data[d].co).length > .00001:
#                         inds.append(d)
#                         cos.append(dat.co.copy())

#                 def_cos.append(cos)
#                 def_inds.append(inds)
#                 sks.append(sk)

#                 # # sk_defs = []
#                 # #     #sk_defs.append( (sk.data[d].co - dat.co).copy() )
#                 # #     sk_defs.append(sk.data[d].co - dat.co)

#                 # for d, dat in enumerate(sk.data):
#                 #     if (ref_defs[d] - dat.co).length < .0001:
#                 #         print()
#                 #         print()
#                 #         print(d)
#                 #         print(ref_sk.name, ref_defs[d])
#                 #         print(sk.name, dat.co)

#                 #         dat.co = act_sk.data[d].co
#                 #     else:
#                 #         dat.co = act_sk.data[d].co

#                 #     # if act_defs[d] is not None:
#                 #     #     # dat.co = ref_sk.data[d].co + act_defs[d] + sk_defs[d]
#                 #     #     dat.co = act_defs[d]
#                 #     # else:
#                 #     #     dat.co = ref_sk.data[d].co

#         while aobj.data.shape_keys.reference_key.name != act_sk.name:
#             bpy.ops.object.shape_key_move(type='UP')

#         for s, sk in enumerate(sks):
#             cos = def_cos[s]
#             inds = def_inds[s]

#             for d, dat in enumerate(sk.data):
#                 dat.co = act_cos[d]

#             for c, co in enumerate(cos):
#                 ind = inds[c]

#                 sk.data[ind].co = co

#         for sk in aobj.data.shape_keys.key_blocks:
#             if sk != act_sk:
#                 sk.relative_key = act_sk

#         return {'FINISHED'}


class SKMP_OT_apply_active_vertex_weights(Operator):
    bl_idname = "skmp.apply_active_vertex_weights"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        key = aobj.active_shape_key

        if aobj.data.shape_keys == None:
            self.report(type={'ERROR'},
                        message='Active Object has no Shape Keys')
            return {'CANCELLED'}

        if key.vertex_group == '':
            self.report(type={'ERROR'},
                        message='Active Shape Key has no Vertex Group')
            return {'CANCELLED'}

        for d, data in enumerate(key.data):
            try:
                weight = aobj.vertex_groups[key.vertex_group].weight(d)
                v_co = aobj.data.vertices[d].co
                s_co = data.co
                data.co = v_co + (s_co-v_co)*weight

            except:
                v_co = aobj.data.vertices[d].co
                data.co = v_co

        key.vertex_group = ''

        return {'FINISHED'}


class SKMP_OT_apply_filtered_vertex_weights(Operator):
    bl_idname = "skmp.apply_filter_vertex_weights"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        if aobj.data.shape_keys == None:
            self.report(type={'ERROR'},
                        message='Active Object has no Shape Keys')
            return {'CANCELLED'}

        names = [key.name for key in aobj.data.shape_keys.key_blocks]
        for name in names:
            key = aobj.data.shape_keys.key_blocks[name]
            if scn_prop.text_filter != '' and scn_prop.text_filter not in key.name or key == aobj.data.shape_keys.reference_key or key.vertex_group == '':
                continue

            for d, data in enumerate(key.data):
                try:
                    weight = aobj.vertex_groups[key.vertex_group].weight(d)
                    v_co = aobj.data.vertices[d].co
                    s_co = data.co
                    data.co = v_co + (s_co-v_co)*weight

                except:
                    v_co = aobj.data.vertices[d].co
                    data.co = v_co

            key.vertex_group = ''

        return {'FINISHED'}


class SKMP_OT_cody_drivers_to_selected_by_name(Operator):
    bl_idname = "skmp.copy_drivers_to_selected_by_name"
    bl_label = ""
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context):
        data = bpy.data
        context = bpy.context

        obs = data.objects
        aobj = context.active_object
        scn = context.scene
        scn_prop = scn.skmp_props

        if aobj.data.shape_keys == None:
            self.report(type={'ERROR'}, message='Object has no Shape Keys')

        names = [sk.name for sk in aobj.data.shape_keys.key_blocks]
        for target in context.selected_objects:
            if target == aobj or target.data.shape_keys == None:
                continue

            # transfer shape key data based on linked data
            for sk in target.data.shape_keys.key_blocks:
                if sk != target.data.shape_keys.reference_key:
                    if sk.name in names:
                        ind = names.index(sk.name)

                        if aobj.data.shape_keys.animation_data != None:
                            copy_drivers(
                                aobj, target, aobj.data.shape_keys.key_blocks[ind], sk)

        return {'FINISHED'}


#
#


_classes = [
    SKMP_OT_rename_filtered_sk,
    SKMP_OT_duplicate_mir_active_sk,
    SKMP_OT_duplicate_mir_filtered_sk,
    SKMP_OT_copy_sel_obs_to_sk,
    SKMP_OT_copy_sks_from_sel_obs,
    SKMP_OT_set_mir_axis,
    SKMP_OT_assign_vertex_group_filtered,
    SKMP_OT_toggle_visibility,
    SKMP_OT_toggle_visibility_filter,
    SKMP_OT_copy_sks_from_sel_mismatching,
    SKMP_OT_explode_sks,
    SKMP_OT_refresh_relations,
    SKMP_OT_apply_shape_keys,
    SKMP_OT_set_active_as_basis,
    SKMP_OT_apply_active_vertex_weights,
    SKMP_OT_apply_filtered_vertex_weights,
    SKMP_OT_cody_drivers_to_selected_by_name,
]


_register, _unregister = register_classes_factory(_classes)


def register():
    _register()
    return


def unregister():
    _unregister()
    return
