#!/usr/bin/python3
# copyright (c) 2018- polygoniq xyz s.r.o.

# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import os
import sys
import blf
import bpy_extras.view3d_utils
import collections
import mathutils
import typing
import bpy
import tempfile
import logging
import logging.handlers
logger_formatter = logging.Formatter(
    "P%(process)d:%(asctime)s:%(name)s:%(levelname)s: %(message)s", "%H:%M:%S")
logger = logging.getLogger(__name__)
try:
    logger.setLevel(int(os.environ.get("POLYGONIQ_LOG_LEVEL", "20")))
except (ValueError, TypeError):
    logger.setLevel(20)
logger.propagate = False
logger_stream_handler = logging.StreamHandler()
logger_stream_handler.setFormatter(logger_formatter)
logger.addHandler(logger_stream_handler)
try:
    log_path = os.path.join(tempfile.gettempdir(), "polygoniq_logs")
    os.makedirs(log_path, exist_ok=True)
    logger_handler = logging.handlers.TimedRotatingFileHandler(
        os.path.join(log_path, f"{__name__}.txt"),
        when="h",
        interval=1,
        backupCount=2,
        utc=True
    )
    logger_handler.setFormatter(logger_formatter)
    logger.addHandler(logger_handler)
except:
    logger.exception(f"Can't create rotating log handler for \"{__name__}\"")
logger.info(f"Logger for \"{__name__}\" initialized in \"{__file__}\" -----")


ADDITIONAL_DEPS_DIR = os.path.abspath(os.path.join(os.path.dirname(__file__), "python_deps"))
try:
    if os.path.isdir(ADDITIONAL_DEPS_DIR) and ADDITIONAL_DEPS_DIR not in sys.path:
        sys.path.insert(0, ADDITIONAL_DEPS_DIR)

    if "polib" not in locals():
        import polib
        from . import preferences
        from . import utils
        from . import image_sizer
        from . import object_render_estimator
        from . import memory_usage
    else:
        import importlib
        polib = importlib.reload(polib)
        preferences = importlib.reload(preferences)
        utils = importlib.reload(utils)
        image_sizer = importlib.reload(image_sizer)
        object_render_estimator = importlib.reload(object_render_estimator)
        memory_usage = importlib.reload(memory_usage)

finally:
    if ADDITIONAL_DEPS_DIR in sys.path:
        sys.path.remove(ADDITIONAL_DEPS_DIR)


bl_info = {
    "name": "memsaver_personal",
    "author": "polygoniq xyz s.r.o.",
    "version": (1, 0, 0),  # bump doc_url as well!
    "blender": (2, 93, 0),
    "location": "memsaver panel in the polygoniq tab in the sidebar of the 3D View window",
    "description": "",
    "category": "System",
    "doc_url": "https://docs.polygoniq.com/memsaver/1.0.0/",
    "tracker_url": "https://polygoniq.com/discord"
}
telemetry = polib.get_telemetry("memsaver")
telemetry.report_addon(bl_info, __file__)


ADDON_CLASSES: typing.List[typing.Type] = []


class ImageSizerOperatorBase(bpy.types.Operator):
    @staticmethod
    def get_target_images(context: bpy.types.Context) -> typing.Set[bpy.types.Image]:
        operator_target = preferences.get_preferences(context).operator_target

        if operator_target == 'SELECTED_OBJECTS':
            return {image for obj in context.selected_objects for image in utils.get_images_used_in_object(obj)}
        elif operator_target == 'SCENE_OBJECTS':
            return {image for obj in context.scene.objects for image in utils.get_images_used_in_object(obj)}
        elif operator_target == 'ALL_OBJECTS':
            return {image for obj in bpy.data.objects for image in utils.get_images_used_in_object(obj)}
        elif operator_target == 'ALL_IMAGES':
            return set(bpy.data.images)
        else:
            raise ValueError(f"Unknown selection target '{operator_target}'")

    def invoke(self, context: bpy.types.Context, event: bpy.types.Event):
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context: bpy.types.Context) -> None:
        prefs = preferences.get_preferences(context)
        self.layout.prop(prefs, "operator_target", text="")


@polib.log_helpers.logged_operator
class ChangeImageSize(ImageSizerOperatorBase):
    bl_idname = "memsaver.change_image_size"
    bl_label = "Change Image Size"
    bl_description = "Change images of given objects, generate lower resolution images on demand " \
        "if necessary"
    bl_options = {'REGISTER'}

    def draw(self, context: bpy.types.Context):
        super().draw(context)

        prefs = preferences.get_preferences(context)
        self.layout.prop(prefs, "change_size_desired_size")
        if prefs.change_size_desired_size == 'CUSTOM':
            self.layout.prop(prefs, "change_size_custom_size")

    def execute(self, context: bpy.types.Context):
        prefs = preferences.get_preferences(context)
        cache_path = prefs.get_cache_path()

        images = ChangeImageSize.get_target_images(context)
        logger.info(f"Working with target images: {images}")
        logger.info(f"desired_size={prefs.change_size_desired_size}")
        logger.info(f"custom_size={prefs.change_size_custom_size}")
        context.window_manager.progress_begin(0, len(images))
        progress = 0

        desired_size = prefs.change_size_custom_size if prefs.change_size_desired_size == 'CUSTOM' else int(
            prefs.change_size_desired_size)
        for image in images:
            try:
                image_sizer.change_image_size(cache_path, image, desired_size)
            except:
                logger.exception(f"Uncaught exception while changing size of image {image.name}")
                self.report(
                    {'WARNING'},
                    f"Errors encountered when changing size of image {image.name}, skipping..."
                )

            progress += 1
            context.window_manager.progress_update(progress)

        context.window_manager.progress_end()
        return {'FINISHED'}


ADDON_CLASSES.append(ChangeImageSize)


@polib.log_helpers.logged_operator
class AdaptiveImageSize(ImageSizerOperatorBase):
    bl_idname = "memsaver.adaptive_image_size"
    bl_label = "Adaptive Image Size"
    bl_description = "Change image size of given objects based on how large the objects appear " \
        "in the render based on active camera, generate lower resolution images if necessary"
    bl_options = {'REGISTER'}

    def draw(self, context: bpy.types.Context):
        super().draw(context)

        prefs = preferences.get_preferences(context)

        self.layout.prop(prefs, "adaptive_quality_factor")
        self.layout.prop(prefs, "adaptive_minimum_size")
        self.layout.prop(prefs, "adaptive_maximum_size")

    @staticmethod
    def get_target_objects(context: bpy.types.Context) -> typing.Iterable[bpy.types.Object]:
        operator_target = preferences.get_preferences(context).operator_target

        if operator_target == 'SELECTED_OBJECTS':
            return context.selected_objects
        elif operator_target == 'SCENE_OBJECTS':
            return context.scene.objects
        elif operator_target == 'ALL_OBJECTS':
            return bpy.data.objects
        elif operator_target == 'ALL_IMAGES':
            # ALL_IMAGES doesn't make sense for this operator, we do the same thing as ALL_OBJECTS
            return bpy.data.objects
        else:
            raise ValueError(f"Unknown selection target '{operator_target}'")

    @classmethod
    def poll(cls, context: bpy.types.Context):
        return context.scene is not None and context.scene.camera is not None

    def execute(self, context: bpy.types.Context):
        prefs = preferences.get_preferences(context)
        cache_path = prefs.get_cache_path()

        objects = list(AdaptiveImageSize.get_target_objects(context))
        logger.info(f"Working with target objects: {objects}")
        logger.info(f"quality_factor={prefs.adaptive_quality_factor}")
        logger.info(f"minimum_size={prefs.adaptive_minimum_size}")
        logger.info(f"maximum_size={prefs.adaptive_maximum_size}")

        context.window_manager.progress_begin(0, 1 + len(objects))
        image_size_map = object_render_estimator.get_size_map_for_objects(
            context.scene,
            context.scene.camera,
            objects,
            prefs.adaptive_quality_factor,
            prefs.adaptive_minimum_size,
            prefs.adaptive_maximum_size,
            True
        )
        progress = 1
        context.window_manager.progress_update(progress)

        for image, change_size_desired_size in image_size_map.items():
            try:
                image_sizer.change_image_size(cache_path, image, change_size_desired_size)
            except:
                logger.exception(f"Uncaught exception while changing size of image {image.name}")
                self.report(
                    {'WARNING'},
                    f"Errors encountered when changing size of image {image.name}, skipping..."
                )
            progress += 1
            context.window_manager.progress_update(progress)

        context.window_manager.progress_end()
        return {'FINISHED'}


ADDON_CLASSES.append(AdaptiveImageSize)


@polib.log_helpers.logged_operator
class RevertImagesToOriginals(ImageSizerOperatorBase):
    bl_idname = "memsaver.revert_images_to_originals"
    bl_label = "Revert Images to Originals"
    bl_description = "Change given images back to their originals. This does not delete lower " \
        "resolution images that may have been generated previously"
    bl_options = {'REGISTER'}

    def execute(self, context: bpy.types.Context):
        images = RevertImagesToOriginals.get_target_images(context)
        logger.info(f"Working with target images: {images}")

        for image in images:
            try:
                image_sizer.revert_to_original(image)
            except Exception as e:
                logger.exception(
                    f"Uncaught exception while reverting image {image.name} to original")
                self.report(
                    {'WARNING'},
                    f"Errors encountered when reverting image {image.name} to original, skipping..."
                )

        return {'FINISHED'}


ADDON_CLASSES.append(RevertImagesToOriginals)


@polib.log_helpers.logged_operator
class CheckDerivatives(ImageSizerOperatorBase):
    bl_idname = "memsaver.check_derivatives"
    bl_label = "Check & Regenerate Derivatives"
    bl_description = "Check that given images all have valid paths, if the path is invalid and " \
        "it is a lower resolution derivative we re-generate it"
    bl_options = {'REGISTER'}

    def execute(self, context: bpy.types.Context):
        prefs = preferences.get_preferences(context)
        cache_path = prefs.get_cache_path()
        logger.debug(f"Cache path: {cache_path}")
        images = CheckDerivatives.get_target_images(context)
        logger.info(f"Working with target images: {images}")

        for image in images:
            try:
                # Unlike in the post_load handler, here we insist on the currently set cache_path
                image_sizer.check_derivative(cache_path, image)
            except Exception as e:
                logger.exception(f"Uncaught exception while checking image {image.name}")
                self.report(
                    {'WARNING'},
                    f"Errors encountered when checking image {image.name}, skipping..."
                )

        return {'FINISHED'}


ADDON_CLASSES.append(CheckDerivatives)


def text_3d(
    world_pos: mathutils.Vector,
    region: bpy.types.Region,
    rv3d: bpy.types.RegionView3D,
    rows: typing.Iterable[str],
    font_id: int = 0,
    font_size: int = 16,
    color: typing.Tuple[float, float, float, float] = (1, 1, 1, 1),
    dpi: int = 72,
) -> None:
    pos_2d = bpy_extras.view3d_utils.location_3d_to_region_2d(
        region,
        rv3d,
        world_pos,
        default=(0, 0, 0)
    )

    for i, row in enumerate(rows):
        blf.position(font_id, pos_2d[0], pos_2d[1] + i * font_size, 0)
        blf.size(font_id, font_size, dpi)
        blf.color(font_id, *color)
        blf.draw(font_id, str(row))


@polib.log_helpers.logged_operator
class PreviewAdaptiveSize(bpy.types.Operator):
    bl_idname = "memsaver.adaptive_size_preview"
    bl_label = "Preview Adaptive Size"
    bl_description = "Starts a preview mode to observe what image sizes will be generated when " \
        "using Adaptive Image Resize operator"
    bl_options = {'REGISTER'}

    bgl_2d_handler_ref = None
    is_running: bool = False
    obj_image_info_map = collections.defaultdict(list)

    def draw(self, context: bpy.types.Context):
        self.layout.label(text="Preview sizes will be generated based on those settings")
        prefs = preferences.get_preferences(bpy.context)
        self.layout.prop(prefs, "adaptive_quality_factor")
        self.layout.prop(prefs, "adaptive_minimum_size")
        self.layout.prop(prefs, "adaptive_maximum_size")

    def __init__(self):
        PreviewAdaptiveSize.bgl_2d_handler_ref = bpy.types.SpaceView3D.draw_handler_add(
            self.draw_px, (), 'WINDOW', 'POST_PIXEL')

    def __del__(self):
        # It is necessary to cancel the operator using __del__, because VSCode reloads
        # don't end the operator if it is running.
        self.cancel(bpy.context)

    def draw_px(self):
        prefs = preferences.get_preferences(bpy.context)
        font_size = prefs.overlay_text_size_px * bpy.context.preferences.view.ui_scale
        region = bpy.context.region
        rv3d = bpy.context.space_data.region_3d
        for obj in bpy.context.selected_objects:
            images = PreviewAdaptiveSize.obj_image_info_map.get(obj, None)
            if images is None:
                continue

            text_3d(
                obj.location, region, rv3d,
                [self._format_image_size(*i) for i in images],
                font_size=font_size,
                color=prefs.overlay_text_color
            )

    def modal(self, context: bpy.types.Context, event: bpy.types.Event):
        for area in context.window.screen.areas:
            if area.type == 'VIEW_3D':
                area.tag_redraw()

        if (event.value == 'PRESS' and event.type == 'ESC'):
            self.cancel(context)
            return {'FINISHED'}

        return {'PASS_THROUGH'}

    def cancel(self, context: bpy.types.Context):
        cls = type(self)
        if hasattr(cls, "bgl_2d_handler_ref") \
           and cls.bgl_2d_handler_ref is not None:
            bpy.types.SpaceView3D.draw_handler_remove(
                cls.bgl_2d_handler_ref, 'WINDOW')
            cls.bgl_2d_handler_ref = None

        cls.is_running = False

    def execute(self, context: bpy.types.Context):
        cls = type(self)
        cls.obj_image_info_map.clear()
        prefs = preferences.get_preferences(context)
        # Pre-compute the maps from all objects in the .blend file, then in draw_px display only
        # information about selected objects
        image_size_map = object_render_estimator.get_size_map_for_objects(
            context.scene,
            context.scene.camera,
            bpy.data.objects,
            prefs.adaptive_quality_factor,
            prefs.adaptive_minimum_size,
            prefs.adaptive_maximum_size,
            True
        )

        for obj in bpy.data.objects:
            images = utils.get_images_used_in_object(obj)
            for img in images:
                orig_size = max(img.size[0], img.size[1])
                new_size = image_size_map.get(img, 1)
                cls.obj_image_info_map[obj].append((img, orig_size, new_size, new_size / orig_size))

            cls.obj_image_info_map[obj].sort(key=lambda x: x[3], reverse=True)

        cls.is_running = True
        context.window_manager.modal_handler_add(self)
        return {'RUNNING_MODAL'}

    def invoke(self, context: bpy.types.Context, event: bpy.types.Event) -> None:
        # If the operator is already running, don't do anything
        if PreviewAdaptiveSize.is_running:
            return {'PASS_THROUGH'}

        return context.window_manager.invoke_props_dialog(self)

    def _format_image_size(
        self,
        img: bpy.types.Image,
        orig_size: int,
        new_size: int,
        ratio: float
    ) -> str:
        return f"{img.name}, {orig_size}px -> {new_size}px, {ratio * 100.0:.0f}%"


ADDON_CLASSES.append(PreviewAdaptiveSize)


@polib.log_helpers.logged_panel
class MemSaverPanel(bpy.types.Panel):
    bl_idname = "VIEW3D_PT_memsaver"
    bl_label = str(bl_info.get("name", "memsaver")).replace("_", " ")
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "polygoniq"
    bl_order = 10 + 1

    def draw_header(self, context: bpy.types.Context):
        try:
            self.layout.label(
                text="", icon_value=polib.ui.icon_manager.get_icon_id("logo_memsaver"))
        except KeyError:
            pass

    def draw_header_preset(self, context: bpy.types.Context) -> None:
        self.layout.operator(preferences.OpenCacheFolder.bl_idname, icon='FILE_CACHE', text="")

    def draw_preview_mode(self, context: bpy.types.Context):
        layout = self.layout
        col = layout.column(align=True)
        col.label(
            text="Select object to preview",
            icon='RESTRICT_SELECT_OFF')

        prefs = preferences.get_preferences(context)
        col.prop(prefs, "overlay_text_size_px")
        col.prop(prefs, "overlay_text_color", text="")
        col.separator()
        row = col.row()
        row.alert = True
        row.label(text="Press ESC to exit", icon='PANEL_CLOSE')

    def draw(self, context: bpy.types.Context):
        layout = self.layout
        if PreviewAdaptiveSize.is_running:
            self.draw_preview_mode(context)
            return

        row = layout.row(align=True)
        row.scale_x = row.scale_y = 1.4
        row.operator(
            AdaptiveImageSize.bl_idname,
            text="Adaptive Image Resize",
            icon='VIEW_PERSPECTIVE'
        )
        row.operator(
            PreviewAdaptiveSize.bl_idname,
            text="",
            icon='VIEWZOOM'
        )

        row = layout.row(align=True)
        row.scale_x = row.scale_y = 1.2
        layout.operator(ChangeImageSize.bl_idname, text="Resize Images", icon='ARROW_LEFTRIGHT')

        col = layout.column(align=True)
        col.operator(RevertImagesToOriginals.bl_idname, icon='LOOP_BACK')
        col.operator(CheckDerivatives.bl_idname)
        col.separator()
        col.operator(memory_usage.EstimateMemoryUsage.bl_idname, icon='MEMORY')


ADDON_CLASSES.append(MemSaverPanel)


@bpy.app.handlers.persistent
def memsaver_load_post(_) -> None:
    """Go through all bpy.data.images and check derivatives, regen if necessary

    Whenever a .blend is loaded we have to go through all images and check that their derivatives
    are present where they should be, if they are not we will regenerate. This can also be achieved
    manually with the "Check & Regenerate" button/operator from the panel.
    """

    logger.info(
        f"Checking all bpy.data.images' derivatives as part of a load_post handler "
        f"(bpy.data.filepath=\"{bpy.data.filepath}\")..."
    )
    for image in bpy.data.images:
        # We purposefully infer cache_path from the image.filepath to avoid regenerating everything
        # when preferences get corrupted or changed
        try:
            image_sizer.check_derivative(None, image)
        except:
            logger.exception(f"Uncaught exception while checking image {image.name}")


def register():
    preferences.register()
    memory_usage.register()

    for cls in ADDON_CLASSES:
        bpy.utils.register_class(cls)

    bpy.app.handlers.load_post.append(memsaver_load_post)


def unregister():
    bpy.app.handlers.load_post.remove(memsaver_load_post)

    for cls in reversed(ADDON_CLASSES):
        bpy.utils.unregister_class(cls)

    memory_usage.unregister()
    preferences.unregister()
