# copyright (c) 2018- polygoniq xyz s.r.o.

import typing
import bpy
import os
import hashlib
import logging
logger = logging.getLogger(__name__)


if "polib" not in locals():
    import polib
    from . import derivative_generator
else:
    import importlib
    polib = importlib.reload(polib)
    derivative_generator = importlib.reload(derivative_generator)


ORIGINAL_PATH_PROPERTY_NAME = "memsaver_original_path"
DERIVATIVE_SIZE_PROPERTY_NAME = "memsaver_derivative_size"


def get_filepath_hash(abs_path: str) -> str:
    """Given an absolute file path, return a hash

    This hash is the base for image derivative paths.
    """

    assert os.path.isabs(abs_path)
    return hashlib.sha256(abs_path.encode("utf-8")).hexdigest()


def get_original_path(image: bpy.types.Image) -> typing.Optional[str]:
    original_path = image.get(ORIGINAL_PATH_PROPERTY_NAME, None)
    if original_path is not None:
        assert isinstance(original_path, str)
        return typing.cast(str, original_path)
    return None


def ensure_original_path(image: bpy.types.Image) -> str:
    original_path = get_original_path(image)
    if original_path in {None, ""}:
        # we don't absolutize the path, we will do that only just before we have to
        image[ORIGINAL_PATH_PROPERTY_NAME] = image.filepath
    else:
        return typing.cast(str, original_path)

    original_path = get_original_path(image)
    # it can be "" in case the image is generated or packed
    assert original_path is not None
    return typing.cast(str, original_path)


def revert_to_original(image: bpy.types.Image) -> None:
    original_path = get_original_path(image)
    if original_path not in {None, ""}:
        # we purposefully don't absolutize the path, we revert back to original,
        # be it relative or absolute
        image.filepath = original_path
        image[DERIVATIVE_SIZE_PROPERTY_NAME] = 0
        logger.debug(
            f"Reverted image {image.name} to the original at {original_path} (bpy path).")
    else:
        logger.debug(
            f"Asked to revert image {image.name} to the original but it already is, "
            f"filepath: {image.filepath}.")


def get_derivative_path(cache_path: str, original_abs_path: str, side_size: int) -> str:
    # TODO: We generate the hash from absolute path, is that OK?
    path_hash = get_filepath_hash(original_abs_path)
    _, ext = os.path.splitext(original_abs_path)
    return os.path.join(cache_path, f"{path_hash}_{side_size}{ext}")


def is_derivative_stale(original_abs_path: str, derivative_path: str) -> bool:
    """Returns True if derivative image does not exist or the original image has changed
    """
    if not os.path.isfile(derivative_path):
        logger.debug(
            f"Derivative at path {derivative_path} is stale because it doesn't exist!")
        return True  # doesn't exist, must be stale

    # derivative has been modified earlier than original, must be stale
    if os.path.getmtime(derivative_path) <= os.path.getmtime(original_abs_path):
        logger.debug(
            f"Derivative at path {derivative_path} is stale because the original "
            f"{original_abs_path} has a newer modified date!")
        return True

    logger.debug(
        f"Derivative at path {derivative_path} is up to date, original at {original_abs_path}.")
    return False


def change_image_size(cache_path: str, image: bpy.types.Image, side_size: int) -> None:
    assert side_size > 0
    original_path = ensure_original_path(image)
    if original_path == "":
        logger.debug(
            f"Image {image.name} has empty original path, it's most likely generated or packed. "
            f"Can't change its size."
        )
        return

    if image.get(DERIVATIVE_SIZE_PROPERTY_NAME, 0) == 0:
        # if the image has the original path and its size is smaller than what's requested, we can
        # do an early out optimization here
        if max(image.size) <= side_size:
            logger.debug(
                f"Image {image.name} has original size {list(image.size)} which is smaller or "
                f"equal to the requested side size {side_size}. Skipping!"
            )
            return

    original_abs_path = os.path.abspath(
        bpy.path.abspath(original_path, library=image.library))
    derivative_path = get_derivative_path(cache_path, original_abs_path, side_size)

    if is_derivative_stale(original_abs_path, derivative_path):
        if os.path.isfile(derivative_path):
            os.unlink(derivative_path)
        derivative_generated: bool = derivative_generator.generate_derivative(
            original_abs_path,
            derivative_path,
            side_size
        )
        if not derivative_generated:
            # we failed to generate the derivative, which means we could be stuck on some weird
            # derivative from the past, e.g. 1x1, switch to original as a safer option
            revert_to_original(image)
            return

    image.filepath = derivative_path
    image[DERIVATIVE_SIZE_PROPERTY_NAME] = side_size
    logger.debug(
        f"Set image {image.name} to derivative {derivative_path} of size {side_size}, "
        f"original at {original_abs_path}."
    )


def check_derivative(cache_path: typing.Optional[str], image: bpy.types.Image) -> None:
    """Re-generate derivative image if the original image file was changed on disk

    cache_path may be None, in this case it will be inferred from derivative absolute path
    """

    original_path = get_original_path(image)
    if original_path is None:  # it's original or original path custom property is not present
        return

    side_size: int = image.get(DERIVATIVE_SIZE_PROPERTY_NAME, 0)
    if side_size == 0:  # it's original or no custom property is present
        return

    # we can infer cache_path, this is useful especially in the post_load handler
    if cache_path is None:
        cache_path = os.path.dirname(image.filepath)

    original_abs_path = os.path.abspath(bpy.path.abspath(original_path, library=image.library))
    derivative_path = get_derivative_path(cache_path, original_abs_path, side_size)
    # This assert is too harsh and would cause user errors if cache_path is changed
    # assert image.filepath == derivative_path
    if is_derivative_stale(original_abs_path, derivative_path):
        if os.path.isfile(derivative_path):
            os.unlink(derivative_path)
        derivative_generated: bool = derivative_generator.generate_derivative(
            original_abs_path,
            derivative_path,
            side_size
        )
        if not derivative_generated:
            return  # TODO: log error?

        image.filepath = derivative_path
        logger.debug(
            f"Re-generated derivative {derivative_path} of size {side_size} from original "
            f"{original_path} because it was stale.")

    else:
        logger.debug(
            f"Derivative {derivative_path} of size {side_size} from original {original_path} is "
            f"up to date, no need to re-generate.")
