# copyright (c) 2018- polygoniq xyz s.r.o.

import typing
import bpy
import os
import tempfile


if "polib" not in locals():
    import polib
else:
    import importlib
    polib = importlib.reload(polib)


telemetry = polib.get_telemetry("memsaver")


MODULE_CLASSES: typing.List[typing.Type] = []


@polib.log_helpers.logged_operator
class OpenLogFolder(bpy.types.Operator):
    bl_idname = "memsaver.open_log_folder"
    bl_label = "Open Log Folder (Debug Information)"
    bl_description = "Opens the directory with logs from all polygoniq addons"
    bl_options = {'REGISTER'}

    def execute(self, context):
        log_path = os.path.join(tempfile.gettempdir(), "polygoniq_logs")
        os.makedirs(log_path, exist_ok=True)
        with open(os.path.join(log_path, "latest_telemetry.txt"), "w") as f:
            f.write(telemetry.dump())
        polib.utils.xdg_open_file(log_path)
        return {'FINISHED'}


MODULE_CLASSES.append(OpenLogFolder)


@polib.log_helpers.logged_operator
class OpenCacheFolder(bpy.types.Operator):
    bl_idname = "memsaver.open_cache_folder"
    bl_label = "Open Cache Folder"
    bl_description = "Opens the directory with cached derivative images"
    bl_options = {'REGISTER'}

    def execute(self, context):
        prefs = get_preferences(context)
        polib.utils.xdg_open_file(prefs.cache_path)
        return {'FINISHED'}


MODULE_CLASSES.append(OpenCacheFolder)


@polib.log_helpers.logged_preferences
class Preferences(bpy.types.AddonPreferences):
    bl_idname = __package__

    cache_path: bpy.props.StringProperty(
        name="Cache Path",
        default=os.path.expanduser("~/memsaver_cache"),
        description="Where derivatives (of various sizes) of images will be cached.",
        subtype='DIR_PATH',
        update=lambda self, context: polib.utils.absolutize_preferences_path(
            self, context, "cache_path")
    )

    # Used to choose target of image sizer operator
    operator_target: bpy.props.EnumProperty(
        name="Target",
        description="Choose to what objects/images the operator should apply to",
        items=[
            ('SELECTED_OBJECTS', "Selected Objects", "All selected objects"),
            ('SCENE_OBJECTS', "Scene Objects", "All objects in current scene"),
            ('ALL_OBJECTS', "All Objects", "All objects in the .blend file"),
            ('ALL_IMAGES', "All Images", "All images, even those not used in any objects")
        ],
        default='SCENE_OBJECTS',
    )

    adaptive_quality_factor: bpy.props.FloatProperty(
        name="Quality Factor",
        description="Object 2D bounds are multiplied by this to figure out image side size. \n\n"
        "Texture sizes are calculated based on how many pixels the objects that use them take "
        "either horizontally or vertically in the camera view. For example if an object is bigger "
        "vertically and it takes 10% of a FHD resolution then it takes 108 px and so its textures "
        "(if not used anywhere else) have to be at least 256px if the quality factor is set to "
        "2.00. Please note that if a big texture is mapped onto an object in a way that only "
        "a fraction of it is actually used, the downscaling might have a detrimental effect since "
        "even though the object is let's say 970 px wide it uses only a small part of the entire "
        "texture which then gets downscaled based on the bounds and not based on actual mapping "
        "density",
        default=2.0,
    )

    adaptive_minimum_size: bpy.props.IntProperty(
        name="Minimum Image Size",
        description="Minimal image size, the algorithm will not choose a smaller size than this",
        default=1,
    )

    adaptive_maximum_size: bpy.props.IntProperty(
        name="Maximum Image Size",
        description="Maximal image size, the algorithm will not choose a larger size than this",
        default=16384,  # 16k
    )

    overlay_text_size_px: bpy.props.FloatProperty(
        name="Overlay Text Size",
        description="Size of the overlay text in pixels",
        default=16.0,
        min=1.0
    )

    overlay_text_color: bpy.props.FloatVectorProperty(
        name="Overlay Text Color",
        description="Color of the text overlay. "
        "Useful when the default value would blend with background",
        min=0.0,
        max=1.0,
        default=(1.0, 1.0, 1.0, 1.0),
        size=4,
        subtype='COLOR'
    )

    change_size_desired_size: bpy.props.EnumProperty(
        name="Desired Size",
        description="Desired Maximum Side Size of Textures",
        items=(
            ("32", "32", "32"),
            ("64", "64", "64"),
            ("128", "128", "128"),
            ("256", "256", "256"),
            ("512", "512", "512"),
            ("1024", "1024", "1024"),
            ("2048", "2048", "2048"),
            ("4096", "4096", "4096"),
            ("8192", "8192", "8192"),
            ("CUSTOM", "CUSTOM", "CUSTOM")
        ),
        default="2048"
    )

    change_size_custom_size: bpy.props.IntProperty(
        name="Custom Size",
        description="Desired Maximum Side Size of Textures",
        default=3072,
    )

    def get_cache_path(self) -> str:
        if not os.path.isdir(self.cache_path):
            os.makedirs(self.cache_path)
        return self.cache_path

    def draw(self, context: bpy.types.Context):
        row = self.layout.row()
        row.prop(self, "cache_path")
        row = self.layout.row()
        row.operator(OpenCacheFolder.bl_idname, icon='FILE_CACHE')

        row = self.layout.row()
        row.operator(OpenLogFolder.bl_idname, icon='EXPERIMENTAL')

        polib.ui.draw_settings_footer(self.layout)


MODULE_CLASSES.append(Preferences)


def get_preferences(context: bpy.types.Context) -> Preferences:
    return context.preferences.addons[__package__].preferences


def register():
    for cls in MODULE_CLASSES:
        bpy.utils.register_class(cls)


def unregister():
    for cls in reversed(MODULE_CLASSES):
        bpy.utils.unregister_class(cls)
