#!/usr/bin/python3
# copyright (c) 2018- polygoniq xyz s.r.o.

import bpy
import os
import tempfile
import logging
import logging.handlers
logger_formatter = logging.Formatter(
    "P%(process)d:%(asctime)s:%(name)s:%(levelname)s: %(message)s", "%H:%M:%S")
logger = logging.getLogger(__name__)
try:
    logger.setLevel(int(os.environ.get("POLYGONIQ_LOG_LEVEL", "20")))
except (ValueError, TypeError):
    logger.setLevel(20)
logger.propagate = False
logger_stream_handler = logging.StreamHandler()
logger_stream_handler.setFormatter(logger_formatter)
logger.addHandler(logger_stream_handler)
try:
    log_path = os.path.join(tempfile.gettempdir(), "polygoniq_logs")
    os.makedirs(log_path, exist_ok=True)
    logger_handler = logging.handlers.TimedRotatingFileHandler(
        os.path.join(log_path, f"{__name__}.txt"),
        when="h",
        interval=1,
        backupCount=2,
        utc=True
    )
    logger_handler.setFormatter(logger_formatter)
    logger.addHandler(logger_handler)
except:
    logger.exception(f"Can't create rotating log handler for \"{__name__}\"")
logger.info(f"Logger for \"{__name__}\" initialized in \"{__file__}\" -----")


if "asset_addon" not in locals():
    from . import asset_addon
    from . import asset_pack
    from . import bl_info_utils
    from . import installation_utils
    from . import linalg
    from . import log_helpers
    from . import material_addon
    from . import megaddon_entry
    from . import node_utils
    from . import rigs_shared
    from . import snap_to_ground
    from . import telemetry_module as telemetry_native_module
    from . import ui
    from . import utils

else:
    import importlib
    asset_addon = importlib.reload(asset_addon)
    asset_pack = importlib.reload(asset_pack)
    bl_info_utils = importlib.reload(bl_info_utils)
    installation_utils = importlib.reload(installation_utils)
    linalg = importlib.reload(linalg)
    log_helpers = importlib.reload(log_helpers)
    material_addon = importlib.reload(material_addon)
    megaddon_entry = importlib.reload(megaddon_entry)
    node_utils = importlib.reload(node_utils)
    rigs_shared = importlib.reload(rigs_shared)
    snap_to_ground = importlib.reload(snap_to_ground)
    telemetry_native_module = importlib.reload(telemetry_native_module)
    ui = importlib.reload(ui)
    utils = importlib.reload(utils)


# fake bl_info so that this gets picked up by vscode blender integration
bl_info = {
    "name": "polib",
    "description": "",
}


def init_polygoniq_global():
    global telemetry_module

    if not hasattr(bpy, "polygoniq_global"):
        bpy.polygoniq_global = {
            "telemetry": {},  # deprecated!
            "telemetry_module": {}
        }

    if not hasattr(bpy.polygoniq_global, "telemetry_module"):
        bpy.polygoniq_global["telemetry_module"] = {}

    # another polygoniq addon might have already initialized telemetry!
    # we want to use just one instance unless it's a different API version
    if telemetry_native_module.API_VERSION in bpy.polygoniq_global["telemetry_module"]:
        telemetry_module = bpy.polygoniq_global["telemetry_module"][telemetry_native_module.API_VERSION]
    else:
        telemetry_module = telemetry_native_module
        bpy.polygoniq_global["telemetry_module"][telemetry_native_module.API_VERSION] = telemetry_module
        telemetry_module.bootstrap_telemetry()


init_polygoniq_global()


def get_telemetry(product: str):
    return telemetry_module.get_telemetry(product)


__all__ = [
    "asset_addon",
    # asset_browser_indexer intentionally missing because it's an executable, not to be imported
    "asset_pack",
    "bl_info_utils",
    "get_telemetry",
    "installation_utils",
    "linalg",
    "log_helpers",
    "material_addon",
    "megaddon_entry",
    "node_utils",
    "rigs_shared",
    "snap_to_ground",
    # telemetry_module intentionally missing, you should interact with it via get_telemetry
    "ui",
    "utils"
]
