#!/usr/bin/python3
# copyright (c) 2018- polygoniq xyz s.r.o.

import bpy
import typing
import logging
logger = logging.getLogger(__name__)


if "asset_addon" not in locals():
    from . import asset_addon
    from . import ui
else:
    import importlib
    asset_addon = importlib.reload(asset_addon)
    ui = importlib.reload(ui)


def find_asset_pack_preferences(pack_name: str) -> typing.Optional[bpy.types.AddonPreferences]:
    for module_name in (pack_name, f"{pack_name}_addon_only", f"{pack_name}_addon"):
        if module_name not in bpy.context.preferences.addons:
            continue

        return bpy.context.preferences.addons[module_name].preferences

    return None


def _register_asset_pack_worker(addon: asset_addon.AssetAddon, pack_name: str, pack_vendor: str) -> typing.Tuple[bool, str]:
    prefs = asset_addon.find_asset_addon_preferences(addon)
    if prefs is None:
        message = f"Is {addon.name} present and enabled in Blender?"
        return False, message

    if not hasattr(prefs, "asset_packs"):
        message = \
            f"Please upgrade {addon.name} to latest version. It seems it is too old to support " \
            f"asset packs"
        return False, message

    for item in prefs.asset_packs:
        if item.name == pack_name:
            logger.info(f"{pack_name} already present in {addon.name} preferences {prefs}")
            # we silently ignore this instead of showing a message
            return True, ""

    item = prefs.asset_packs.add()
    item.name = pack_name
    item.vendor = pack_vendor
    # TODO: Remove this when we migrate to megaddon all
    if hasattr(prefs, "update_asset_provider_bq67"):
        prefs.update_asset_provider_bq67()
    logger.info(f"{pack_name} newly registered with {addon.name} preferences {prefs}")
    return True, ""


def register_asset_pack(addon: asset_addon.AssetAddon, pack_name: str, pack_vendor: str, attempts: int = 0) -> None:
    result, message = _register_asset_pack_worker(addon, pack_name, pack_vendor)
    if result:
        return

    # Asset packs can now be used independently in Blender 3.2 or newer if users integrate them with
    # the Blender's native Asset Browser. Only show the error in Blenders older than 3.2 to avoid
    # spamming legit users
    if bpy.app.version < (3, 2, 0) and attempts == 10:
        logger.error(f"Failed to register {pack_name} as an asset pack for {addon.name}. {message}")
        ui.show_message_box(
            f"Failed to register {pack_name} as an asset pack for {addon.name}. {message}",
            title=f"{pack_name} registration failed",
            icon='ERROR'
        )

    # we try every second for the first 10 seconds, then every 10 seconds
    interval = 1 if attempts < 10 else 10
    # we need to try to register with asset addon for a few attempts
    # this is important in case the user enables the asset pack before enabling the asset addon
    bpy.app.timers.register(
        lambda: register_asset_pack(addon, pack_name, pack_vendor, attempts + 1),
        first_interval=interval
    )


def unregister_asset_pack(addon: asset_addon.AssetAddon, pack_name: str) -> bool:
    prefs = asset_addon.find_asset_addon_preferences(addon)
    if prefs is None:
        return False

    for i, item in enumerate(prefs.asset_packs):
        if item.name == pack_name:
            prefs.asset_packs.remove(i)
            logger.info(f"{pack_name} unregistered from {addon.name} preferences {prefs}")
            return True

    # The pack could already be unregistered from before or from vscode calling unregister() twice
    return False
