"""
Module designed to act as a link between Blender and RizomUV 2022.2+.
"""

import platform

import bpy
from bpy.props import PointerProperty
from bpy.types import WindowManager

from rizomuv_bridge_link import preferences, properties
from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.utility import resolve_ruv_path
from rizomuv_bridge_link.gui import root_panel
from rizomuv_bridge_link.gui.addon_settings_tab import (
    advanced_settings_subpanel,
    core_settings_subpanel,
)
from rizomuv_bridge_link.gui.live_remote_tab import remote_actions_subpanel, rizom_settings_subpanel
from rizomuv_bridge_link.gui.operations_tab import (
    export_settings_subpanel,
    import_settings_subpanel,
)
from rizomuv_bridge_link.operators import (
    remote_action,
    reset_rizom_settings,
    rizom_automatic,
    rizom_export,
    rizom_import,
    rizom_load,
    rizom_quit,
    validate_uvmaps,
)
from rizomuv_bridge_link.preferences import RizomUVBridgePreferences

bl_info = {
    "name": "RizomUV Bridge - Link Edition",
    "description": "Work seamlessly between Blender and RizomUV.",
    "author": "Matt Ashpole",
    "version": (1, 0, 1),
    "blender": (3, 3, 0),
    "location": "View3D > Sidebar",
    "doc_url": "https://blendermarket.com/products/rizomuv-bridge/docs",
    "tracker_url": "https://blendermarket.com/products/rizomuv-bridge",
    "category": "Import-Export",
}

if platform.system() != "Windows":
    raise OSError(
        "This version of the addon is only compatible with Windows until the official Rizom Python module is updated."
    )

PROPERTIES = (preferences.RizomUVBridgePreferences, properties.RizomUVBridgeGUITabs, properties.RizomUVBridgeUVMaps)
OPERATORS = (
    rizom_export.ExportToRizom,
    rizom_import.ImportFromRizom,
    rizom_automatic.AutomaticRizom,
    validate_uvmaps.ValidateUVMaps,
    rizom_load.RizomLoad,
    rizom_quit.RizomQuit,
    reset_rizom_settings.ResetAddonPreferencesRizom,
    remote_action.ExecuteActionRizom,
)
LAYOUTS = (
    root_panel.RizomUVBridgePanel,
    export_settings_subpanel.RizomUVExportSettingsPanel,
    import_settings_subpanel.RizomUVImportSettingsPanel,
    remote_actions_subpanel.RizomUVRemoteActionsPanel,
    rizom_settings_subpanel.RizomUVSettingsPanel,
    core_settings_subpanel.RizomUVActionsCoreSettingsPanel,
    advanced_settings_subpanel.RizomUVActionsAdvancedSettingsPanel,
)


def register():
    """Register operators, menus and properties when addon is enabled."""

    for item in PROPERTIES:
        bpy.utils.register_class(item)

    prefs: "RizomUVBridgePreferences" = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences

    for item in OPERATORS:
        bpy.utils.register_class(item)

    for item in LAYOUTS:
        item.bl_category = prefs.category_name
        bpy.utils.register_class(item)

    WindowManager.RizomUVBridgeGUITabs = PointerProperty(type=properties.RizomUVBridgeGUITabs)
    WindowManager.RizomUVBridgeUVMaps = PointerProperty(type=properties.RizomUVBridgeUVMaps)


def unregister():
    """Unregister operators, menus and properties when addon is disabled."""

    for item in PROPERTIES:
        bpy.utils.unregister_class(item)

    for item in OPERATORS:
        bpy.utils.unregister_class(item)

    for item in LAYOUTS:
        bpy.utils.unregister_class(item)

    del WindowManager.RizomUVBridgeGUITabs
    del WindowManager.RizomUVBridgeUVMaps
