"""Functions for handling object selection in Blender."""

import bpy
from bpy.types import Object


def get_selected() -> tuple[Object, list[Object]]:
    """Return both the active object and other selected mesh objects.

    If there are no selected objects, the active object is used instead,
    this is because it is possible to have an unselected active object.

    Returns:
        The active object and a list of selected objects.
    """

    act_obj = bpy.context.active_object
    sel_objs = [obj for obj in bpy.context.selected_objects if obj.type == "MESH"]
    if not sel_objs:
        act_obj.select_set(True)
        sel_objs = [act_obj]

    return act_obj, sel_objs


def get_clones(objs: list[Object]) -> list[Object]:
    """Finds objects with shared data blocks.

    Args:
        objs: List of objects to test against each-other.

    Returns:
        List of objects with shared data blocks.
    """

    data_blocks = []
    clone_objects = []

    for obj in objs:
        if obj.data not in data_blocks:
            data_blocks.append(obj.data)
        else:
            clone_objects.append(obj)

    return clone_objects


def change_selection(deselect_objs: list[Object] | bool, select_objs: list[Object] | None = None) -> None:
    """Change object selections.

    Args:
        deselect_objs: List of objects to deselect OR boolean to deselect all objects or no objects.
        select_objs: List of objects to select.
    """

    if isinstance(deselect_objs, list):
        for obj in deselect_objs:
            obj.select_set(False)

    elif deselect_objs is True:
        bpy.ops.object.select_all(action="DESELECT")

    if select_objs:
        for obj in select_objs:
            obj.select_set(True)
