"""Functions for validating object and UV map suitability for export."""

from bpy.types import Object


def validate_object_uvmaps(obj: Object) -> str | bool:
    """Runs tests on object UV maps.
    Test cases:
        - No uvmaps
        - illegal characters in name ('.')

    Args:
        obj: Object to test.

    Returns:
        True if all tests pass or else a string containing error information.
    """
    uv_maps = obj.data.uv_layers

    if not uv_maps:
        return f"<{obj.name}> has no UV maps"

    for uv_map in uv_maps:
        if "." in uv_map.name:
            return f"<{obj.name} : {uv_map.name}> contains illegal character <.>"

    return True


def compare_object_uvmaps(control_obj: Object, test_objs: list[Object]) -> str | bool:
    """
    Compares the UV maps of each object in the test group against the
    control object to make sure all UV maps are named the same and none are
    missing.

    Args:
        control_obj: Already tested object to compare others with.
        test_objs: Objects to test.

    Returns:
        True if all tests pass or else a string containing error information.
    """
    control_names = [uv_map.name for uv_map in control_obj.data.uv_layers]

    for obj in test_objs:
        if not obj.data.uv_layers:
            return obj.name

        test_uvmap_names = [uv_map.name for uv_map in obj.data.uv_layers]
        if control_names != test_uvmap_names:
            return f"<{obj.name}> UV maps do not match those of <{control_obj.name}>"

    return True


def validate_object_names(obj: Object, fix: bool) -> str | bool:
    """Tests the name of the given object.
    Test cases:
        - illegal characters in name ('.').

    Args:
        obj: The object to test.
        fix: Automatically fix the name problem.

    Returns:
        True if all tests pass or else a string containing error information.
    """

    if "." in obj.name:
        if fix:
            obj.name = obj.name.replace(".", "_")
            return True
        return f"<{obj.name}> includes an illegal character <.>, either replace it or export using FBX format"

    return True


def compare_mesh_data(objects: tuple[Object, Object]) -> bool:
    """Compare mesh data of two objects and make sure they match.

    Args:
        objects: Object pair to compare.

    Returns:
        True if objects have equal geometry.
    """
    return len(objects[0].data.polygons) == len(objects[1].data.polygons)
