"""Functions handling object/layer visibility in viewport."""

from bpy.types import LayerCollection, Object

import bpy


def unhide_objects(search_objects: list[Object]) -> list[Object]:
    """Reveal any of the given objects that are hidden.

    Args:
        search_objects: Objects to search for and reveal.

    Returns:
        List of objects that were hidden.
    """
    hidden_objs = [obj for obj in search_objects if not obj.visible_get()]

    for obj in hidden_objs:
        obj.hide_set(False)

    return hidden_objs


def unhide_layers(search_objects: list[Object]) -> tuple[list[LayerCollection], list[LayerCollection]]:
    """Reveal hidden and excluded collections if they contain any of the given objects.

    Args:
        search_objects: Objects to search collections for.

    Returns:
        List of collections that were originally excluded.
        List of collections that were originally hidden.
    """
    layers_list = list(bpy.context.view_layer.layer_collection.children)

    # Recursively append child layers.
    for layer in layers_list:
        for child in layer.children:
            layers_list.append(child)

    obj_layers = [layer for layer in layers_list if any(obj in search_objects for obj in layer.collection.objects)]

    excluded_layers = [layer for layer in obj_layers if layer.exclude is True]
    hidden_layers = [layer for layer in obj_layers if layer.hide_viewport is True]

    for layer in excluded_layers:
        layer.exclude = False
    for layer in hidden_layers:
        layer.hide_viewport = False

    return excluded_layers, hidden_layers


def hide_layers(layers_hide: list[LayerCollection], layers_exclude: list[LayerCollection]) -> None:
    """Hide or exclude the given layers.

    Args:
        layers_hide: Layers to hide.
        layers_exclude: Layers to exclude.
    """
    for layer in layers_hide:
        layer.hide_viewport = True

    for layer in layers_exclude:
        layer.exclude = True


def hide_objects(objects_hide: list[Object]) -> None:
    """Hide given objects.

    Args:
        objects_hide:
    """
    for obj in objects_hide:
        obj.hide_set(True)
