"""Format and print logs information."""


def print_bad_geometry_report(bad_geo_objs: list[str]) -> None:
    """Format and print a report of all objects that had geometry not matching the object in the scene.

    Args:
        bad_geo_objs: Dictionary containing each object name and a list of UV maps that were missing.
    """
    column1_header = "Incorrect Geometry"

    column1_width = len(max(bad_geo_objs, key=len))
    if column1_width < len(column1_header):
        column1_width = len(column1_header)

    print("INCORRECT GEOMETRY - FINAL REPORT\n")
    print(
        "The following objects did not have vertex indices matching the original object in the scene, "
        "this means that the UV data will not be able to transfer.\n"
        "This can occasionally happen due to problems with the Blender FBX export plugin.\n"
        "You can get your UVs into the scene by using the 'Replace Objects' import option."
    )
    print("\n")
    print(f"{column1_header:{column1_width}}")
    print(column1_width * "─")

    for name in bad_geo_objs:
        print(f"{name:{column1_width}}")

    print(column1_width * "─")


def print_updated_uvmaps_report(updated_uvmaps: dict[str, list[str]]) -> None:
    """Format and print a report of all updated UV maps.

    Args:
        updated_uvmaps: Dictionary containing each object name and a list of UV maps updated.
    """

    column1_header = "Objects"
    column2_header = "Updated UV Maps"
    column_seperator = " │ "

    column1_width = len(max(updated_uvmaps.keys(), key=len))
    if column1_width < 11:
        column1_width = 11

    divider_width = sum((len(column1_header), len(column2_header), len(column_seperator), column1_width))

    print("UPDATED UV MAPS - FINAL REPORT\n")
    print("UV Maps were transferred from the source object.\n")
    print(f"{column1_header:{column1_width}}", end="")
    print(f"{column_seperator}", end="")
    print(f"{column2_header}")

    print(divider_width * "─")

    for obj_name, uv_maps in updated_uvmaps.items():
        print(f"{obj_name:{column1_width}}", end="")
        print(f"{column_seperator}", end="")
        print(f", ".join(uv_maps))

    print(divider_width * "─")


def print_missing_uvmaps_report(missing_uvmaps: dict[str, list[str]]) -> None:
    """Format and print a report of all UV maps that should have been updated, but were unable to be located.

    Args:
        missing_uvmaps: Dictionary containing each object name and a list of UV maps that were missing.
    """

    column1_header = "Objects"
    column2_header = "Missing UV Maps"
    column_seperator = " │ "

    column1_width = len(max(missing_uvmaps.keys(), key=len))
    if column1_width < 11:
        column1_width = 11

    divider_width = sum((len(column1_header), len(column2_header), len(column_seperator), column1_width))

    print("MISSING UV MAPS - FINAL REPORT\n")
    print(
        "Some UV Maps were not able to find a match on the paired object, "
        "matching UV channels were automatically generated."
    )
    print(f"{column1_header:{column1_width}}", end="")
    print(f"{column_seperator}", end="")
    print(f"{column2_header}")

    print(divider_width * "─")

    for obj_name, uv_maps in missing_uvmaps.items():
        print(f"{obj_name:{column1_width}}", end="")
        print(f"{column_seperator}", end="")
        print(f", ".join(uv_maps))

    print(divider_width * "─")


def print_missing_objects_report(missing_objs: list[str]) -> None:
    """Format and print a report of all objects that had a match imported but were unable to be located.

    Args:
        missing_objs: List of missing object names.
    """

    column1_header = "Missing Objects"
    column1_width = len(column1_header)

    print("MISSING OBJECTS - FINAL REPORT\n")
    print("Some imported objects were unable to find a matching object in the scene.\n")

    print(f"{column1_header:{column1_width}}")

    print(column1_width * "─")

    print("\n".join(missing_objs))

    print(column1_width * "─")
