"""Handle actions that can be executed in Rizom."""

import functools
from typing import TYPE_CHECKING

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.utility import misc, units

if TYPE_CHECKING:
    from RizomUVLink import CRizomUVLink
    from rizomuv_bridge_link.preferences import RizomUVBridgePreferences


def repeatable(func):
    """
    Check if apply to all UV maps is enabled.
    If it is then loop the action across all UV maps.
    """

    @functools.wraps(func)
    def wrapper(self):
        """Function wrapper."""
        if self.action_source == "EXPORT":
            apply_to_all = self.prefs.apply_to_all_uvmaps_export
        else:
            apply_to_all = self.prefs.apply_to_all_uvmaps_action

        if apply_to_all:
            for uvmap in self.uv_maps:
                self.link.Uvset({"Mode": "SetCurrent", "Name": uvmap})
                func(self)
        else:
            func(self)

    return wrapper


class RizomActionsManager:
    """Action manager."""

    def __init__(
        self,
        link: "CRizomUVLink",
        prefs: "RizomUVBridgePreferences",
        uv_maps: list[str],
        active_uv_map: str,
        action_source: str,
    ):
        """
        Args:
            link: Instance of the link between Rizom and Blender.
            prefs: Addon preferences access.
            uv_maps: List of UV map names inside Rizom.
            active_uv_map: The current UV map name in Rizom.
            action_source: Source of the command ("EXPORT", "REMOTE").
        """

        self.link = link
        self.prefs = prefs
        self.uv_maps = uv_maps
        self.active_uv_map = active_uv_map
        self.file_map = {"FBX": str(constants.FBX_PATH), "OBJ": str(constants.OBJ_PATH)}
        self.action_source = action_source

        if self.action_source == "EXPORT":
            self.action = self.prefs.export_actions
        elif self.action_source == "REMOTE":
            self.action = self.prefs.remote_actions

    def __do_autoseams_select(self) -> None:
        """Build correct autoseams code."""

        self.__autoseams_settings()

        params = {
            "PrimType": "Edge",
            "WorkingSet": "Visible&UnLocked",
            "IslandGroupMode": "Group",
            "Select": True,
            "ResetBefore": True,
            "ProtectMapName": "Protect",
            "FilterIslandVisible": True,
            "Auto": {
                "PipesCutter": self.link.Get("Vars.AutoSelect.CutHandles"),
                "HandleCutter": self.link.Get("Vars.AutoSelect.LinkHoles"),
                "QuadLoopCutter": self.link.Get("Vars.AutoSelect.OpenCylinders"),
                "StretchLimiter": self.link.Get("Vars.AutoSelect.Bijectiver"),
                "Quality": self.link.Get("Vars.AutoSelect.BijectiverMinQ"),
                "StoreCoordsUVW": True,
                "FlatteningUnfoldParams": {
                    "BorderIntersections": self.link.Get("Prefs.OverlapsOn"),
                    "TriangleFlips": self.link.Get("Prefs.TriangleFlipsOn"),
                },
            },
        }

        self.__edit_autoseams_params(params, self.action)
        self.link.Select(params)

    def __edit_autoseams_params(self, params: dict, action: str) -> None:
        """Edit the values of the parameter dictionary based on selected action.

        Args:
            params: The selection command parameters.
            action: The action that needs to be setup.
        """
        autoseams_dict = params["Auto"]

        match action:
            case "AUTOSEAMS_SHARP_EDGES":
                autoseams_dict["SharpEdges"] = {"AngleMin": self.prefs.sharp_angle}

            case "AUTOSEAMS_MOSAIC":
                autoseams_dict["QuasiDevelopable"] = {
                    "Developability": self.prefs.mosaic_force,
                    "IslandPolyNBMin": 1,
                    "FitCones": False,
                    "Straighten": True,
                }

            case "AUTOSEAMS_PELT":
                variables = (self.prefs.pelt_trunk, self.prefs.pelt_branch, self.prefs.pelt_leaf)
                seg_levels = []
                for i, pref in enumerate(variables, start=1):
                    if pref:
                        seg_levels.append(i)
                autoseams_dict["Skeleton"] = {"Open": True, "SegLevels": seg_levels}

    def __autoseams_settings(self) -> None:
        """Setup autoseams settings based on user preferences.

        Returns
            The requested code block.
        """
        match self.action:
            case "AUTOSEAMS_SHARP_EDGES":
                self.link.Set({"Path": "Vars.AutoSelect.SharpEdges.Angle", "Value": self.prefs.sharp_angle})
            case "AUTOSEAMS_MOSAIC":
                self.link.Set({"Path": "Vars.AutoSelect.Mosaic.Developability", "Value": self.prefs.mosaic_force})
            case "AUTOSEAMS_PELT":
                self.link.Set({"Path": "Vars.AutoSelect.Hierarchical.Branches", "Value": self.prefs.pelt_branch})
                self.link.Set({"Path": "Vars.AutoSelect.Hierarchical.Leafs", "Value": self.prefs.pelt_leaf})
                self.link.Set({"Path": "Vars.AutoSelect.Hierarchical.Trunk", "Value": self.prefs.pelt_trunk})

        stretch_limiter = False
        if self.prefs.stretch_quality > 0:
            stretch_limiter = True

        self.link.Set({"Path": "Vars.AutoSelect.Bijectiver", "Value": stretch_limiter})
        self.link.Set({"Path": "Vars.AutoSelect.BijectiverMinQ", "Value": self.prefs.stretch_quality})
        self.link.Set({"Path": "Vars.AutoSelect.CutHandles", "Value": self.prefs.cut_handles})
        self.link.Set({"Path": "Vars.AutoSelect.LinkHoles", "Value": self.prefs.link_gaps})
        self.link.Set({"Path": "Vars.AutoSelect.OpenCylinders", "Value": self.prefs.open_cylinders})

    @repeatable
    def action_offset_groups(self):
        """Offset groups for baking."""

        def offset_group(group_path: str) -> None:
            """Offset all except one of the islands contained in the given group.

            Args:
                group_path: Path to the group.
            """
            island_ids = self.link.Get(f"{group_path}.IslandIDs")
            del island_ids[0]

            self.link.Select(
                {
                    "PrimType": "Island",
                    "WorkingSet": "Visible",
                    "IslandGroupMode": "Group",
                    "Select": True,
                    "ResetBefore": True,
                    "IDs": island_ids,
                    "List": True,
                }
            )
            self.link.Move(
                {
                    "WorkingSet": "Visible&Selected",
                    "PrimType": "Island",
                    "Geometrical": "Transform",
                    "Transform": [1, 0, self.prefs.group_offset_value, 0, 1, 0, 0, 0, 1],
                }
            )

        root_tiles = []

        for child in self.link.Get("Lib.Mesh.RootGroup.Children"):
            if self.link.Get(f"Lib.Mesh.RootGroup.Children.{child}.IsTile"):
                root_tiles.append(child)

        for tile in root_tiles:
            for group in self.link.Get(f"Lib.Mesh.RootGroup.Children.{tile}.Children"):
                if self.link.Get(f"Lib.Mesh.RootGroup.Children.{tile}.Children.{group}.Properties.Selected"):
                    group_path = f"Lib.Mesh.RootGroup.Children.{tile}.Children.{group}"
                    offset_group(group_path)

    def action_set_uvmap(self, name: str) -> None:
        """Set chosen UV map as active.

        Args:
            name: Name of the UV map to set as active.
        """
        self.link.Uvset({"Mode": "SetCurrent", "Name": name})

    @repeatable
    def action_reset_uvmap(self, target_map: str = None) -> None:
        """Weld all UV boundaries.

        Args:
            target_map: The name of the UV map to be reset.
        """
        if target_map is None:
            target_map = self.link.Get("Lib.CurrentUVSetName")

        unique_name = misc.find_unique_name(self.link.Get("Lib.UVSets").keys())

        self.link.Uvset({"Mode": "Create", "Name": unique_name})
        self.link.Uvset({"Mode": "SetCurrent", "Name": unique_name})

        self.link.Uvset({"Mode": "Delete", "Name": target_map})
        self.link.Uvset({"Mode": "Create", "Name": target_map})

        self.link.Uvset({"Mode": "SetCurrent", "Name": target_map})
        self.link.Uvset({"Mode": "Delete", "Name": unique_name})

        self.link.ResetTo3d({"WorkingSet": "Visible", "Rescale": True})

    @repeatable
    def action_autoseams(self) -> None:
        """Execute autoseams script."""

        self.__do_autoseams_select()

        self.link.Cut({"PrimType": "Edge", "WorkingSet": "Visible&UnLocked"})
        self.link.Load({"Data": {"CoordsUVWInternalPath": "#Mesh.Tmp.AutoSelect.UVW"}})
        self.link.IslandGroups({"Mode": "DistributeInTilesByBBox", "WorkingSet": "Visible"})
        self.link.IslandGroups(
            {
                "Mode": "DistributeTilesContent",
                "WorkingSet": "Visible&UnLocked",
                "FreezeIslands": True,
                "UseTileLocks": True,
                "UseIslandLocks": True,
                "GroupPaths": ["RootGroup"],
            }
        )
        self.link.Pack(
            {
                "RootGroup": "RootGroup",
                "WorkingSet": "Visible",
                "ProcessTileSelection": False,
                "RecursionDepth": 1,
                "Translate": True,
                "AuxGroup": "RootGroup",
                "LayoutScalingMode": 0,
            }
        )

    @repeatable
    def action_apply_settings(self):
        """Set Rizom settings from the addon UI."""

        # region Unwrap Settings
        self.link.Set({"Path": "Prefs.TriangleFlipsOn", "Value": self.prefs.prevent_flips})
        self.link.Set({"Path": "Prefs.OverlapsOn", "Value": self.prefs.prevent_overlaps})
        # endregion

        # region Packing Settings
        self.link.IslandGroups(
            {
                "Mode": "SetGroupsProperties",
                "WorkingSet": "Visible",
                "GroupPaths": ["RootGroup"],
                "Properties": {"Pack": {"MarginSize": (self.prefs.margin * (1 / self.prefs.map_res))}},
            }
        )
        self.link.IslandGroups(
            {
                "Mode": "SetGroupsProperties",
                "WorkingSet": "Visible",
                "GroupPaths": ["RootGroup"],
                "Properties": {"Pack": {"PaddingSize": (self.prefs.padding * (1 / self.prefs.map_res))}},
            }
        )
        self.link.IslandGroups(
            {
                "Mode": "SetGroupsProperties",
                "WorkingSet": "Visible",
                "GroupPath": "RootGroup",
                "Properties": {"Pack": {"MapResolution": self.prefs.map_res}},
            }
        )
        self.link.IslandGroups(
            {
                "Mode": "SetGroupsProperties",
                "WorkingSet": "Visible",
                "GroupPath": "RootGroup",
                "Properties": {
                    "Pack": {"Scaling": {"TexelDensity": units.texel_density(self.prefs.td_target, self.prefs.td_unit)}}
                },
            }
        )
        self.link.Set(
            {
                "Path": "Lib.Scene.Settings.TexelDensityDisplayUnit",
                "Value": f"tx/{self.prefs.td_unit}",
                "UndoAble": True,
            }
        )

        self.link.IslandGroups(
            {
                "Mode": "SetGroupsProperties",
                "WorkingSet": "Visible",
                "GroupPaths": ["RootGroup"],
                "Properties": {"Pack": {"Rotate": {"Mode": int(self.prefs.orientation)}}},
            }
        )

        self.link.IslandGroups(
            {
                "Mode": "SetGroupsProperties",
                "WorkingSet": "Visible",
                "GroupPaths": ["RootGroup"],
                "Properties": {"Pack": {"Rotate": {"Step": self.prefs.step_angle}}},
            }
        )

        # endregion

    def action_load(self, file_type: str, active_uv: str) -> None:
        """Load objects in Rizom and set the active UV map.

        Args:
            file_type: File format being loaded. ["OBJ", "FBX"]
            active_uv: The name of the active UV map.
        """

        params = {
            "File.Path": self.file_map[file_type],
            "File.XYZUVW": True,
            "File.UVWProps": True,
            "File.ImportGroups": True,
            "File.Normals": True,
            "__Focus": True,
        }
        self.link.Load(params)
        if file_type == "OBJ":
            self.link.Uvset({"Mode": "Rename", "Name": active_uv})

        self.link.Uvset({"Mode": "SetCurrent", "Name": active_uv})
