"""Handle the RizomUV live link and module."""

import sys
from pathlib import Path
from typing import TYPE_CHECKING

from rizomuv_bridge_link.data import constants

if TYPE_CHECKING:
    from RizomUVLink import CRizomUVLink


class RizomUVLinkInstance:
    """Store the RizomUVLink."""

    link: "CRizomUVLink" = None


class RizomUVLinkManager:
    """Handles the link to RizomUV. Establishes connection and returns data."""

    def __init__(self, rizom_path: str):
        """
        Args:
            rizom_path: Path to the RizomUV executable.
        """
        self.rizom_path = rizom_path
        self.__append_module()
        import RizomUVLink

        self.RizomUVLink = RizomUVLink

        self.link: "CRizomUVLink" = RizomUVLinkInstance.link
        self.port = None
        self.__setup_link()

    def __append_module(self) -> None:
        """
        Add RizomUVLink module to the path,
        so it is accessible in this class.
        """
        link_module = Path(self.rizom_path).parent / "RizomUVLink"
        sys.path.append(str(link_module))

    def __setup_link(self) -> None:
        """
        Check if a link to Rizom exists in the current session
        if not it will create a new link.
        """
        if self.link is None:
            RizomUVLinkInstance.link = self.RizomUVLink.CRizomUVLink()

        self.link = RizomUVLinkInstance.link

    def get_rizom_version(self) -> int:
        """
        Get the version of RizomUV linked to the instance
        and convert it into an integer for easy comparison.

        Returns:
            Int representing RizomUV version.
        """
        string_list = self.link.RizomUVVersion().split(".")
        del string_list[-1]
        return int("".join(string_list))

    def get_objects(self) -> list[str]:
        """
        Get object names from RizomUV.
        Remove DefaultObject as it is not a real object.

        Returns:
            The names of the objects in Rizom.
        """
        objects: list[str] = self.link.Get("Lib.Mesh.ObjectsNames")
        if "DefaultObject" in objects:
            objects.remove("DefaultObject")
        return objects

    def get_current_uvmap(self) -> str:
        """Get the name of the current UV Map in RizomUV.

        Returns:
            The name of the UVMap.
        """
        return self.link.Get("Lib.CurrentUVSetName")

    def __check_connection(self) -> bool:
        """Check if the connection to Rizom is live.

        Returns
            True if connection is live.
        """

        if self.link.TCPPortIsOpen(self.port):
            try:
                self.link.RizomUVVersion()
                return True
            except self.RizomUVLink.CZEx:
                return False

    def connect_to_rizom(self) -> bool:
        """Attempt to connect to Rizom using stored port number.

        Returns:
            True if connection is achieved.
        """
        with open(constants.PORT, "r", encoding="utf-8") as f:
            port = f.read()

        if port is not None or "None":
            try:
                self.port = int(port)
            except ValueError:
                return False

        self.link.Connect(self.port)
        return self.__check_connection()

    def run_rizom(self) -> None:
        """
        Run Rizom if it is not already open.
        Save port to external txt file, so it can be used by other Blender instances.
        """

        if self.connect_to_rizom():
            return

        RizomUVLinkInstance.port = self.link.RunRizomUV(self.rizom_path)

        self.port = RizomUVLinkInstance.port
        with open(constants.PORT, "w", encoding="utf-8") as f:
            f.write(str(self.port))

    def get_uvmaps(self) -> tuple[str, dict]:
        """Get UV maps in the scene.

        Returns:
            Name of the current UV map.
            Dictionary where the keys are the names of the UV maps.
        """

        return self.link.Get("Lib.CurrentUVSetName"), self.link.Get("Lib.UVSets")
