"""Sub panel that docks in under the live-link tab."""

import bpy
from bpy.types import Context, UILayout

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.gui.operations_tab.export_settings_subpanel import ActionSettingsUIGroups, actions_dic
from rizomuv_bridge_link.preferences import RizomUVBridgePreferences


class RizomUVRemoteActionsPanel(bpy.types.Panel):
    """RizomUV remote actions panel."""

    bl_idname = "PANEL_PT_RizomUVBridgeRemoteActions"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "RizomUV"
    bl_context = "objectmode"
    bl_label = "RizomUV Live Remote"
    bl_parent_id = "PANEL_PT_RizomUVBridge"

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences

    @classmethod
    def poll(cls, context):
        """Check given condition to see if class should be available.

        Args:
            context (Context): Current windowmanager and data context.

        Returns:
            Bool: Check the RIZOM_LIVE UI tab is selected.
        """
        return context.window_manager.RizomUVBridgeGUITabs.tabs == "RIZOM_LIVE"

    def draw(self, context: Context) -> None:
        """Draw UI elements into the panel UI layout.

        Args:
            context: Current data context.

        """
        layout = self.layout

        box = layout.box()
        self.action_options(box)

        if self.prefs.remote_actions in constants.AUTOSEAMS:
            box = layout.box()
            self.common_autoseams_settings(box)
            box = layout.box()
            row = box.row(align=True)
            row.scale_y = 1.3
            row.label(text=f"{actions_dic[self.prefs.remote_actions]} Settings:")
            ActionSettingsUIGroups(self.prefs, box).return_ui(self.prefs.remote_actions)

    def action_options(self, container: UILayout) -> None:
        """The UI group for the action shared settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3
        row.prop(self.prefs, "apply_to_all_uvmaps_export", text="Apply To All UV Maps")
        row.prop(self.prefs, "apply_settings_action", text="Apply Rizom Settings")

        row = container.row(align=True)
        row.scale_y = 1.5
        row.prop(self.prefs, "remote_actions", text="Actions")
        row.operator("ruv_link.rizom_action", text="Execute").settings_only = False

    def common_autoseams_settings(self, container: UILayout):
        """The UI group for common autoseams settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3

        if self.prefs.remote_actions in constants.AUTOSEAMS:
            row.prop(self.prefs, "weld_before_autoseams", text="Weld Before Autoseams", toggle=True)
            row = container.row(align=True)
            row.scale_y = 1.3
            row.prop(self.prefs, "cut_handles", toggle=True)
            row.prop(self.prefs, "link_gaps", toggle=True)
            row.prop(self.prefs, "open_cylinders", toggle=True)

            row = container.row(align=True)
            row.scale_y = 1.3
            row.prop(self.prefs, "stretch_quality")
