"""Sub panel that docks in under the live-link tab."""

import bpy
from bpy.types import Context, UILayout

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.preferences import RizomUVBridgePreferences
from rizomuv_bridge_link.properties import RizomUVBridgeUVMaps


class RizomUVSettingsPanel(bpy.types.Panel):
    """RizomUV layout settings panel."""

    bl_idname = "PANEL_PT_RizomUVBridgeRizomSettings"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "RizomUV"
    bl_context = "objectmode"
    bl_label = "RizomUV Settings"
    bl_parent_id = "PANEL_PT_RizomUVBridge"
    bl_options = {"DEFAULT_CLOSED"}

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences
        self.uvmaps_prop: RizomUVBridgeUVMaps = bpy.context.window_manager.RizomUVBridgeUVMaps

    @classmethod
    def poll(cls, context):
        """Check given condition to see if class should be available.

        Args:
            context (Context): Current windowmanager and data context.

        Returns:
            Bool: Check the RIZOM_LIVE UI tab is selected.
        """
        return context.window_manager.RizomUVBridgeGUITabs.tabs == "RIZOM_LIVE"

    def draw(self, context: Context) -> None:
        """Draw UI elements into the panel UI layout.

        Args:
            context: Current data context.
        """
        layout = self.layout
        box = layout.box()
        row = box.row()
        row.scale_y = 1.3
        row.operator("ruv_link.rizom_action", text="Apply Settings").settings_only = True
        row.operator("ruv_link.reset_prefs", text="Reset Panel")

        box = layout.box()
        box.label(text="Unwrap Settings:")
        self.unwrap_settings(box)

        box = layout.box()
        box.label(text="Packing Settings:")
        self.packing_settings(box)

        box = layout.box()
        box.label(text="Layout Settings:")
        self.layout_settings(box)

    def unwrap_settings(self, container: UILayout) -> None:
        """The UI group for the unwrap settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3

        row.prop(self.prefs, "prevent_flips", toggle=True)
        row.prop(self.prefs, "prevent_overlaps", toggle=True)

    def packing_settings(self, container: UILayout) -> None:
        """The UI group for the packing settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3
        row.prop(self.prefs, "accuracy")
        row.prop(self.prefs, "iterations")

        row = container.row(align=True)
        row.scale_y = 1.3
        row.prop(self.prefs, "orientation", text="")
        row.prop(self.prefs, "step_angle")

    def layout_settings(self, container: UILayout) -> None:
        """The UI group for the layout settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3

        row.prop(self.prefs, "margin")
        row.prop(self.prefs, "padding")

        row = container.row(align=True)
        row.scale_y = 1.3

        row.prop(self.prefs, "map_res")

        row = container.row(align=True)
        row.scale_y = 1.3

        row.prop(self.prefs, "td_target")
        row.prop(self.prefs, "td_unit", text="")

