"""Sub panel that docks in under the operations tab."""

import bpy
from bpy.types import AddonPreferences, Context, UILayout

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.preferences import RizomUVBridgePreferences

actions_dic = {
    "RESET_UVS": "Reset / Weld UVs",
    "AUTOSEAMS_SHARP_EDGES": "Autoseams - Sharp Edges",
    "AUTOSEAMS_MOSAIC": "Autoseams - Mosaic",
    "AUTOSEAMS_PELT": "Autoseams - Pelt",
    "AUTOSEAMS_BOX": "Autoseams - Box",
}


class ActionSettingsUIGroups:
    """
    Contains the UI groups for all scripts.
    It is used to automatically draw the correct group.
    """

    def __init__(self, prefs: AddonPreferences, container: UILayout):
        """
        Args:
            prefs: Addon preferences access.
            container: Container to insert a UI group into.
        """

        self.prefs = prefs
        self.container = container
        self.actions_dict = {
            "RESET_UVS": self.reset_uvs_ui,
            "AUTOSEAMS_SHARP_EDGES": self.sharp_edges_ui,
            "AUTOSEAMS_MOSAIC": self.mosaic_ui,
            "AUTOSEAMS_PELT": self.pelt_ui,
            "OFFSET_FOR_BAKE": self.offset_selected_groups_ui,
        }

    def return_ui(self, key: str) -> callable:
        """Return the requested UI group.

        Args:
             key: Key to the script to retrieve the UI group for.

        Returns:
            The UI method matching the key.
        """

        if key in self.actions_dict:
            return self.actions_dict[key]()
        else:
            return self.no_settings_ui()

    def no_settings_ui(self) -> None:
        """The UI group for scripts which have no unique settings."""

        row = self.container.row()
        row.scale_y = 1.3

        row.label(text="No Settings Available")

    def reset_uvs_ui(self) -> None:
        """The UI group for reset uvs settings."""
        row = self.container.row(align=True)
        row.scale_y = 1.3

        row.prop(self.prefs, "apply_to_all_uvmaps_export", text="Apply To All UV Maps", toggle=True)

    def sharp_edges_ui(self) -> None:
        """The UI group for the Sharp Edges Autoseams script."""

        row = self.container.row()
        row.scale_y = 1.3

        row.prop(self.prefs, "sharp_angle")

    def mosaic_ui(self) -> None:
        """The UI group for the Mosaic Autoseams script."""

        row = self.container.row()
        row.scale_y = 1.3

        row.prop(self.prefs, "mosaic_force")

    def pelt_ui(self) -> None:
        """The UI group for the Pelt Autoseams script."""

        row = self.container.row()
        row.scale_y = 1.3

        row.prop(self.prefs, "pelt_trunk", toggle=True)
        row.prop(self.prefs, "pelt_branch", toggle=True)
        row.prop(self.prefs, "pelt_leaf", toggle=True)

    def offset_selected_groups_ui(self) -> None:
        """The UI group for the baking groups offset script."""

        row = self.container.row()
        row.scale_y = 1.3

        row.prop(self.prefs, "group_offset_direction")
        row.prop(self.prefs, "group_offset_value")


class RizomUVExportSettingsPanel(bpy.types.Panel):
    """Export settings panel."""

    bl_idname = "PANEL_PT_RizomUVBridgeExportSettings"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "RizomUV"
    bl_context = "objectmode"
    bl_label = "Export Settings"
    bl_parent_id = "PANEL_PT_RizomUVBridge"

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences

    @classmethod
    def poll(cls, context):
        """Check given condition to see if class should be available.

        Args:
            context (Context): Current windowmanager and data context.

        Returns:
            Bool: Check the OPERATIONS UI tab is selected.
        """
        return context.window_manager.RizomUVBridgeGUITabs.tabs == "OPERATIONS"

    def draw(self, context: Context) -> None:
        """Draw UI elements into the panel UI layout.

        Args:
            context: Current data context.

        """
        layout = self.layout

        box = layout.box()
        self.export_options(box)

        if self.prefs.export_actions != "EXPORT":
            box = layout.box()
            self.common_autoseams_settings(box)
            if self.prefs.export_actions in constants.AUTOSEAMS:
                box = layout.box()
                row = box.row(align=True)
                row.scale_y = 1.3
                row.label(text=f"{actions_dic[self.prefs.export_actions]} Settings:")
                ActionSettingsUIGroups(self.prefs, box).return_ui(self.prefs.export_actions)

    def export_options(self, container: UILayout) -> None:
        """The UI group for the export settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3
        row.prop(self.prefs, "exclude_clones")
        row.prop(self.prefs, "confirm_export")

        row = container.row(align=True)
        row.scale_y = 1.3
        row.prop(self.prefs, "triangulate", text="Triangulate Meshes")

        split = row.split()
        split.prop(self.prefs, "fix_names", text="Fix Names")
        if self.prefs.file_type != "OBJ":
            split.enabled = False

        row = container.row(align=True)
        row.scale_y = 1.3
        row.prop(self.prefs, "apply_settings_export", text="Apply Rizom Settings")

        box = container.box()
        row = box.row(align=True)
        row.scale_y = 1.5
        row.prop(self.prefs, "export_actions", text="Actions")

    def common_autoseams_settings(self, container: UILayout):
        """The UI group for common action settings.

        Args:
            container: Box UI element to insert a UI group into.
        """
        row = container.row(align=True)
        row.scale_y = 1.3

        split1 = row.split()
        split1.prop(self.prefs, "apply_to_all_uvmaps_export", text="Apply To All UV Maps", toggle=True)
        if self.prefs.file_type != "FBX":
            split1.enabled = False

        split2 = row.split()
        if self.prefs.export_actions in constants.AUTOSEAMS:
            split2.prop(self.prefs, "weld_before_autoseams", text="Weld Before Autoseams", toggle=True)
            row = container.row(align=True)
            row.scale_y = 1.3
            row.prop(self.prefs, "cut_handles", toggle=True)
            row.prop(self.prefs, "link_gaps", toggle=True)
            row.prop(self.prefs, "open_cylinders", toggle=True)

            row = container.row(align=True)
            row.scale_y = 1.3
            row.prop(self.prefs, "stretch_quality")
