"""This is the parent panel that all other elements are built on."""

import bpy
from bpy.types import Context, UILayout

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.preferences import RizomUVBridgePreferences
from rizomuv_bridge_link.properties import RizomUVBridgeGUITabs, RizomUVBridgeUVMaps


class RizomUVBridgePanel(bpy.types.Panel):
    """Addon UI panel."""

    bl_idname = "PANEL_PT_RizomUVBridge"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "RizomUV"
    bl_context = "objectmode"
    bl_label = "RizomUV Bridge - Link Edition"

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences
        self.uvmaps_prop: RizomUVBridgeUVMaps = bpy.context.window_manager.RizomUVBridgeUVMaps
        self.tabs_prop: RizomUVBridgeGUITabs = bpy.context.window_manager.RizomUVBridgeGUITabs

    def draw(self, context: Context) -> None:
        """Draw UI elements into the panel UI layout.

        Args:
            context: Current data context.
        """
        layout = self.layout

        box = layout.box()
        row = box.row(align=True)
        row.scale_y = 1.3

        row.prop(self.tabs_prop, "tabs", icon_only=True, expand=True)

        match self.tabs_prop.tabs:
            case "OPERATIONS":
                row.label(text="UV Transfer")
                self.primary_operations(layout.box())
                self.uv_maps(layout.box())
                self.file_manager(layout.box())

            case "RIZOM_LIVE":
                row.label(text="Live Remote")

            case "ADDON_SETTINGS":
                row.label(text="Addon Settings")

    @staticmethod
    def primary_operations(container: UILayout) -> None:
        """Primary operations UI elements.

        Args:
            container: container layout to place UI elements inside.
        """
        row = container.row(align=True)
        row.label(text="Primary Operations:", icon="UV_DATA")

        row = container.row()
        row.scale_y = 1.3

        row.operator("ruv_link.rizom_export", text="Export", icon="EXPORT")
        row.operator("ruv_link.rizom_import", text="Import", icon="IMPORT")

        row = container.row()
        row.scale_y = 1.3

        row.operator("ruv_link.rizom_automatic", text="Automatic", icon="AUTO")
        row.operator("ruv_link.rizom_quit", text="Close Rizom", icon="QUIT")

    def uv_maps(self, container: UILayout) -> None:
        """UV Map display and tools UI elements.

        Args:
            container: container layout to place UI elements inside.
        """
        row = container.row(align=True)
        row.scale_y = 1.3

        split = row.split(factor=0.65, align=True)
        split.prop(self.uvmaps_prop, "uv_maps")
        split.operator("ruv_link.validate_uvmaps", text="Validate UVSets", icon="UV_DATA")

    def file_manager(self, container: UILayout) -> None:
        """File manager UI elements.

        Args:
            container: container layout to place UI elements inside.
        """
        row = container.row(align=True)
        row.scale_y = 1.3

        split = row.split(factor=0.65, align=True)
        split.prop(self.prefs, "file_type")
        row = split.row(align=True)
        row.operator("ruv_link.rizom_load", text="Load", icon="FILE_CACHE")
