"""Send commands to RizomUV instance using RizomUVLiveLink module."""

from typing import TYPE_CHECKING

import bpy
from bpy.types import Context

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.rizomuv_link import actions_manager, rizomuv_link_manager
from rizomuv_bridge_link.functions.rizomuv_link.actions_manager import RizomActionsManager
from rizomuv_bridge_link.functions.utility import json_utility

if TYPE_CHECKING:
    from rizomuv_bridge_link.preferences import RizomUVBridgePreferences

json_parser = json_utility.JsonUtility(constants.GUI_STRINGS)


class ExecuteActionRizom(bpy.types.Operator):
    """Execute actions in RizomUV remotely."""

    bl_description = "Execute actions in RizomUV remotely"
    bl_idname = "ruv_link.rizom_action"
    bl_label = "Action (RizomUV)"
    bl_options = {"REGISTER", "INTERNAL"}

    settings_only: bpy.props.BoolProperty(name="Settings")

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences

    @staticmethod
    def action_map(action_manager: RizomActionsManager) -> dict:
        """Map actions to functions to call them.

        Args:
            action_manager: Instance of the action manager.

        Returns:
            Dictionary mapping actions to functions.
        """

        return {
            "RESET_UVS": action_manager.action_reset_uvmap,
            "AUTOSEAMS_SHARP_EDGES": action_manager.action_autoseams,
            "AUTOSEAMS_MOSAIC": action_manager.action_autoseams,
            "AUTOSEAMS_PELT": action_manager.action_autoseams,
            "AUTOSEAMS_BOX": action_manager.action_autoseams,
            "OFFSET_GROUPS": action_manager.action_offset_groups,
        }

    def execute(self, context: Context) -> set[str]:
        """Execute the operator.

        Args:
            context: Current data context.

        Returns:
            Enum in {'RUNNING_MODAL', 'CANCELLED', 'FINISHED', 'PASS_THROUGH'}
        """

        link_manager = rizomuv_link_manager.RizomUVLinkManager(self.prefs.rizom_path)
        if not link_manager.connect_to_rizom():
            self.report({"ERROR"}, "You are not connected to an instance of RizomUV.")
            return {"CANCELLED"}

        act_uv, uv_maps = link_manager.get_uvmaps()
        uv_names = list(uv_maps.keys())

        action_manager = actions_manager.RizomActionsManager(link_manager.link, self.prefs, uv_names, act_uv, "REMOTE")

        if self.settings_only:
            action_manager.action_apply_settings()
            return {"FINISHED"}

        if self.prefs.apply_settings_action:
            action_manager.action_apply_settings()

        self.action_map(action_manager)[self.prefs.remote_actions]()

        return {"FINISHED"}
