"""Automatically run the export operator, perform a quick unwrap then import the UVs."""

from typing import TYPE_CHECKING

import bpy
from bpy.types import Context, Event

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.rizomuv_link import rizomuv_link_manager

if TYPE_CHECKING:
    pass


class AutomaticRizom(bpy.types.Operator):
    """Automatically use Rizom to do a quick unwrap with no user input."""

    bl_description = "Automatically use Rizom to do a quick unwrap with no user input"
    bl_idname = "ruv_link.rizom_automatic"
    bl_label = "Automatic (RizomUV)"
    bl_options = {"REGISTER", "INTERNAL"}

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Tests if the operator is able to run.

        Args:
            context: Current data context.

        Returns:
            True if there is an active object and the export action is not set to standard export.
        """

        return (
            bpy.context.active_object is not None
            and bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences.export_actions != "EXPORT"
        )

    def execute(self, context: Context) -> set[str]:
        """Execute the operator.

        Args:
            context: Current data context.

        Returns:
            Enum in {'RUNNING_MODAL', 'CANCELLED', 'FINISHED', 'PASS_THROUGH'}
        """

        bpy.ops.ruv_link.rizom_export()
        bpy.ops.ruv_link.rizom_import()

        rizomuv_link_manager.RizomUVLinkInstance.link.Quit()

        return {"FINISHED"}

    def invoke(self, context: Context, event: Event) -> callable:
        """
        Invoke the operator, this is called before execution.
        Used to prompt the user for export confirmation as a method
        for avoiding accidental overwrites.

        Args:
            context: Current data context.
            event: Event context.

        Returns:
            window_manager dialog or execute method.
        """
        prefs = context.preferences.addons[constants.PACKAGE_NAME].preferences

        if prefs.confirm_export:
            return context.window_manager.invoke_confirm(self, event)

        return self.execute(context)
