"""Export to Rizom using RizomUVLiveLink module."""

from typing import TYPE_CHECKING

import bpy
from bpy.types import Context, Event, Object

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.object_functions import export, mesh_edit, object_selection as obj_sel, validation
from rizomuv_bridge_link.functions.rizomuv_link import actions_manager, rizomuv_link_manager
from rizomuv_bridge_link.functions.rizomuv_link.rizomuv_link_manager import RizomUVLinkManager

if TYPE_CHECKING:
    from rizomuv_bridge_link.preferences import RizomUVBridgePreferences
    from rizomuv_bridge_link.properties import RizomUVBridgeUVMaps


class ExportToRizom(bpy.types.Operator):
    """Export selected mesh items and open the file in RizomUV."""

    bl_description = "Export selected objects to a file and opens it in RizomUV"
    bl_idname = "ruv_link.rizom_export"
    bl_label = "Export (RizomUV)"
    bl_options = {"REGISTER", "INTERNAL"}

    def __init__(self):
        self.prefs: "RizomUVBridgePreferences" = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences
        self.uvmaps_prop: "RizomUVBridgeUVMaps" = bpy.context.window_manager.RizomUVBridgeUVMaps

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Tests if the operator is able to run.

        Args:
            context: Current data context.

        Returns:
            True if there is an active object.
        """

        return context.active_object is not None and context.active_object.type == "MESH"

    def get_objects(self) -> tuple[Object, list[Object], list[Object]]:
        """Build separate lists of scene objects.

        Returns:
            Active object, selected objects, selected objects with clones removed.
        """
        act_obj, sel_objs = obj_sel.get_selected()

        if self.prefs.exclude_clones:
            filtered_objs = [obj for obj in sel_objs if obj not in obj_sel.get_clones(sel_objs)]
        else:
            filtered_objs = sel_objs

        return act_obj, sel_objs, filtered_objs

    def validate_objects(self, act_obj: Object, export_objs: list[Object]) -> bool:
        """
        Runs tests on objects to make sure they are not in a format that will cause
        problems in RizomUV.

        Args:
            act_obj: The active object.
            export_objs: All the objects chosen for export.

        Returns:
            True if all tests pass.
        """

        result = validation.validate_object_uvmaps(act_obj)
        if result is not True:
            self.report({"ERROR"}, result)
            return False

        # Compare each of the other objects against the active object, making sure UVMaps match.
        result = validation.compare_object_uvmaps(act_obj, export_objs)
        if result is not True:
            self.report({"ERROR"}, result)
            return False

        # Testing the name of each object.
        if self.prefs.file_type == "OBJ":
            for obj in export_objs:
                result = validation.validate_object_names(obj, self.prefs.fix_names)
                if result is not True:
                    self.report({"ERROR"}, result)
                    return False

        return True

    def post_export_actions(self, rizom_link: RizomUVLinkManager, uvmaps: list[str], act_uvmap: str) -> None:
        """Send commands to Rizom performing actions.

        Args:
            rizom_link: Instance of the RizomUVLinkManager.
            uvmaps: List of UV maps on the objects.
            act_uvmap: The target UV map.
        """

        action_manager = actions_manager.RizomActionsManager(rizom_link.link, self.prefs, uvmaps, act_uvmap, "EXPORT")
        action_manager.action_load(self.prefs.file_type, self.uvmaps_prop.uv_maps)

        if self.prefs.apply_settings_export:
            action_manager.action_apply_settings()

        if self.prefs.export_actions != "EXPORT":
            if self.prefs.export_actions in constants.AUTOSEAMS:
                if self.prefs.weld_before_autoseams:
                    action_manager.action_reset_uvmap()
                action_manager.action_autoseams()

            if self.prefs.export_actions == "RESET_UVS":
                action_manager.action_reset_uvmap()

    def execute(self, context: Context) -> set[str]:
        """Execute the operator.

        Args:
            context: Current data context.

        Returns:
            Enum in {'RUNNING_MODAL', 'CANCELLED', 'FINISHED', 'PASS_THROUGH'}
        """
        link_manager = rizomuv_link_manager.RizomUVLinkManager(self.prefs.rizom_path)

        act_obj, _, chosen_objs = self.get_objects()
        if not self.validate_objects(act_obj, chosen_objs):
            return {"CANCELLED"}

        uvmap_set = [uvmap.name for uvmap in act_obj.data.uv_layers]

        local_view = context.space_data.local_view
        if local_view:
            bpy.ops.view3d.localview(frame_selected=False)

        if self.prefs.triangulate:
            mesh_edit.triangulate_objects(chosen_objs)

        export.export_selected(self.prefs.file_type)

        if local_view:
            bpy.ops.view3d.localview(frame_selected=False)

        link_manager.run_rizom()

        self.post_export_actions(link_manager, uvmap_set, self.uvmaps_prop.uv_maps)

        return {"FINISHED"}

    def invoke(self, context: Context, event: Event) -> callable:
        """
        Invoke the operator, this is called before execution.
        Used to prompt the user for export confirmation as a method
        for avoiding accidental overwrites.

        Args:
            context: Current data context.
            event: Event context.

        Returns:
            window_manager dialog or execute method.
        """

        prefs = context.preferences.addons[constants.PACKAGE_NAME].preferences

        if prefs.confirm_export:
            return context.window_manager.invoke_confirm(self, event)

        return self.execute(context)
