"""Import from the transfer files and transfer UV data."""

import bpy
from bpy.types import Context, Object

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.object_functions import file_import, object_selection, validation, visibility
from rizomuv_bridge_link.functions.reports import print_reports
from rizomuv_bridge_link.functions.rizomuv_link import rizomuv_link_manager
from rizomuv_bridge_link.functions.utility import bpy_helper
from rizomuv_bridge_link.preferences import RizomUVBridgePreferences


class ImportFromRizom(bpy.types.Operator):
    """Export selected mesh items and open the file in RizomUV."""

    bl_description = "Import objects from the transfer files and transfer UV data to scene objects by name"
    bl_idname = "ruv_link.rizom_import"
    bl_label = "Import (RizomUV)"
    bl_options = {"REGISTER", "INTERNAL", "UNDO"}

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences
        self.updated_uvmaps = {}
        self.missing_uvmaps = {}
        self.bad_geometry = []
        self.missing_scene_objects = []
        self.current_uvmap = None  # OBJ format resets to default UV name, this is used to remember the name

    def transfer_object_uv(
            self, data_object: Object, transfer_object: Object
    ) -> tuple[list[str], list[str], list[str]]:
        """Transfer UV data from one object to another.

        Args:
            data_object: The object with the UV data.
            transfer_object: The object to transfer data to.

        Returns:
            The names of the UV maps that were updated.
            The names of any UV maps that were missing.
            The names of objects with non-matching geometry.
        """
        bpy.ops.object.select_all(action="DESELECT")
        transfer_object.select_set(True)
        bpy.context.view_layer.objects.active = data_object

        if self.current_uvmap:
            data_object.data.uv_layers[0].name = self.current_uvmap

        uv_layers = data_object.data.uv_layers
        iterations = len(uv_layers)

        transfer_object_uv_layers = transfer_object.data.uv_layers

        if self.prefs.delete_uvmaps and self.prefs.file_type == "FBX":
            while transfer_object.data.uv_layers:
                transfer_object.data.uv_layers.remove(transfer_object.data.uv_layers[0])

        missing_uv_maps = []
        updated_uv_maps = []
        geometry_corrupt = []

        while iterations:
            for uv_map in uv_layers:
                iterations -= 1
                if uv_map.name not in transfer_object_uv_layers:
                    transfer_object.data.uv_layers.new(name=uv_map.name)
                    missing_uv_maps.append(uv_map.name)
                data_object.data.uv_layers.active = data_object.data.uv_layers[uv_map.name]
                transfer_object.data.uv_layers.active = transfer_object.data.uv_layers[uv_map.name]

                bpy.ops.object.join_uvs()

                updated_uv_maps.append(uv_map.name)

                if not validation.compare_mesh_data((data_object, transfer_object)):
                    geometry_corrupt.append(data_object.name)

        return updated_uv_maps, missing_uv_maps, geometry_corrupt

    def loop_uv_transfer(self, transfer_objects: list[Object]) -> None:
        """
        Loop the UV transfer process across all objects
        and log data from the process.

        Args:
            transfer_objects: The objects to loop over with transfer method.
        """

        for obj in transfer_objects:
            try:
                data_obj = bpy.data.objects[obj["rizom_tag"]]
            except KeyError:
                continue

            # Needed since Blender 3.6 to avoid bug with duplicate objects recieving wrong uv map.
            bpy.ops.object.mode_set(mode="EDIT")
            bpy.ops.object.mode_set()

            updated_maps, missing_maps, bad_geometry = self.transfer_object_uv(data_obj, obj)

            if updated_maps:
                self.updated_uvmaps[obj["rizom_tag"]] = updated_maps
            if missing_maps:
                self.missing_uvmaps[obj["rizom_tag"]] = missing_maps
            if bad_geometry:
                self.bad_geometry.extend(bad_geometry)

    @staticmethod
    def mark_boundaries(mark_seams: bool, mark_sharp: bool, objects: list[Object]) -> None:
        """Mark seams and/or sharp edges from UV island boundaries.

        Args:
            mark_seams: Mark UV seams.
            mark_sharp: Mark sharp edges.
            objects: Objects to mark.
        """

        object_selection.change_selection(True, objects)
        bpy.context.view_layer.objects.active = objects[0]
        bpy.ops.object.mode_set(mode="EDIT")

        vert, edge, face = bpy_helper.sel_mode(edge=True)

        bpy.ops.mesh.select_all(action="SELECT")
        bpy.ops.uv.select_all(action="SELECT")

        if mark_sharp:
            bpy.ops.mesh.mark_sharp(clear=True)
        if mark_seams:
            bpy.ops.mesh.mark_seam(clear=True)

        bpy.ops.uv.seams_from_islands(mark_seams=mark_seams, mark_sharp=mark_sharp)
        bpy.ops.mesh.select_all(action="DESELECT")

        bpy_helper.sel_mode(vert, edge, face)
        bpy.ops.object.mode_set()

    def compare_with_rizom_scene(self, rizom_objects: list[str]) -> list[Object]:
        """
        Build a list of objects that have the same name as those loaded in Rizom.
        Apply a tag to these objects keeping a note of their original name then
        rename them so that they do not clash with objects being imported.

        Args:
            rizom_objects: List of object names obtained from Rizom.
        """
        scene_objects = []

        for name in rizom_objects:
            try:
                obj = bpy.data.objects[name]
            except KeyError:
                self.missing_scene_objects.append(name)
                continue

            obj["rizom_tag"] = name
            obj.name = "arbitrary_name"
            scene_objects.append(obj)

        return scene_objects

    def cleanup_scene(self, delete_objs: list[Object], select_objects: list[Object], active_obj: Object) -> None:
        """Cleanup the data created by the bridge.

        Args:
            delete_objs: Objects to delete.
            select_objects: Objects to select.
            active_obj: Object to make active.
        """

        for obj in delete_objs:
            for material in obj.data.materials.items():
                bpy.data.materials.remove(material[1])
            bpy.data.objects.remove(obj, do_unlink=True)

        for obj in select_objects:
            obj.select_set(True)
            if not self.prefs.replace_objects:
                if obj["rizom_tag"]:
                    obj.name = obj["rizom_tag"]
                    del obj["rizom_tag"]

        for mesh in bpy.data.meshes:
            if mesh.users == 0:
                bpy.data.meshes.remove(mesh)

        bpy.context.view_layer.objects.active = active_obj

    def render_report(self) -> None:
        """
        Render a final report in the console.
        Printing all information gathered during the import process.
        """
        if self.prefs.replace_objects:
            self.report({"INFO"}, "Objects were replaced, no more information to report.")
            return

        elif self.bad_geometry:
            self.report(
                {"WARNING"}, "Some objects had bad geometry and will not transfer UVs correctly. See console for logs."
            )

        elif self.updated_uvmaps and not any((self.missing_uvmaps, self.missing_scene_objects, self.bad_geometry)):
            self.report({"INFO"}, "UV Maps updated, no problems encountered. See console for logs.")

        elif self.updated_uvmaps and self.missing_uvmaps:
            self.report(
                {"INFO"},
                "UV Maps updated, some UV maps could not be located"
                " and were automatically generated. See console for logs.",
            )

        elif self.updated_uvmaps and self.missing_scene_objects:
            self.report({"WARNING"}, "UV Maps updated, some some objects could not be located. See console for logs.")

        elif not self.updated_uvmaps:
            self.report({"ERROR"}, "No UV Maps were updated. See console for logs.")

        if self.bad_geometry:
            print("\n")
            print_reports.print_bad_geometry_report(self.bad_geometry)

        if self.updated_uvmaps:
            print("\n")
            print_reports.print_updated_uvmaps_report(self.updated_uvmaps)

        if self.missing_uvmaps:
            print("\n")
            print_reports.print_missing_uvmaps_report(self.missing_uvmaps)

        if self.missing_scene_objects:
            print("\n")
            print_reports.print_missing_objects_report(self.missing_scene_objects)

    def execute(self, context: Context) -> set[str]:
        """Execute the operator.

        Args:
            context: Current data context.

        Returns:
            Enum in {'RUNNING_MODAL', 'CANCELLED', 'FINISHED', 'FINISHED', 'INTERFACE'}
        """
        local_view = context.space_data.local_view
        if local_view:
            bpy.ops.view3d.localview(frame_selected=False)

        link_manager = rizomuv_link_manager.RizomUVLinkManager(self.prefs.rizom_path)
        if not link_manager.connect_to_rizom():
            self.report({"ERROR"}, "You are not connected to an instance of RizomUV.")
            return {"CANCELLED"}

        if self.prefs.file_type == "OBJ":
            self.current_uvmap = link_manager.get_current_uvmap()

        rizom_objects = link_manager.get_objects()
        scene_objects = self.compare_with_rizom_scene(rizom_objects)
        excluded_layers, hidden_layers = visibility.unhide_layers(scene_objects)
        hidden_objects = visibility.unhide_objects(scene_objects)

        loaded_file: str = link_manager.link.Get("Prefs.LastLoadedFile")
        file_type = loaded_file[-3:].upper()
        link_manager.link.Save({"File": {file_type: {"UseUVSetNames": True}, "Path": loaded_file, "UVWProps": True}})

        file_import.import_file(file_type)
        imported_objects = [bpy.data.objects[name] for name in rizom_objects]

        if not self.prefs.replace_objects:
            self.loop_uv_transfer(scene_objects)

        if self.prefs.replace_objects:
            delete_objects = scene_objects
            select_objects = imported_objects
        else:
            delete_objects = imported_objects
            select_objects = scene_objects

        if not select_objects:
            if delete_objects:
                self.cleanup_scene(delete_objs=delete_objects)
                self.render_report()
                return {"CANCELLED"}

        if self.prefs.sharp_angle or self.prefs.mark_seams:
            self.mark_boundaries(self.prefs.mark_seams, self.prefs.mark_sharp, select_objects)

        self.cleanup_scene(delete_objects, select_objects, select_objects[0])

        if not self.prefs.reveal_hidden:
            visibility.hide_layers(hidden_layers, excluded_layers)
            visibility.hide_objects(hidden_objects)

        self.render_report()

        return {"FINISHED"}
