"""
Scan UV object UV Maps for issues that could cause an error in RizomUV.
If any errors are found, fix them.
"""

from typing import TYPE_CHECKING

import bpy
from bpy.types import Context, Object

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.object_functions import object_selection as obj_sel

if TYPE_CHECKING:
    from rizomuv_bridge_link.preferences import RizomUVBridgePreferences
    from rizomuv_bridge_link.properties import RizomUVBridgeUVMaps


class ValidateUVMaps(bpy.types.Operator):
    """Scan selected object UV maps and fix any incompatibility with RizomUV."""

    bl_description = (
        "Scan selected object UV maps and fix any incompatibility with RizomUV."
        "\nWARNING: This is destructive and will reset any problem UV maps"
    )
    bl_idname = "ruv_link.validate_uvmaps"
    bl_label = "Validate UVMaps"
    bl_options = {"REGISTER", "INTERNAL", "UNDO"}

    act_obj_map_changes = {}
    changed_objs = []

    def __init__(self):
        self.prefs: RizomUVBridgePreferences = bpy.context.preferences.addons[constants.PACKAGE_NAME].preferences
        self.uvmaps_prop: RizomUVBridgeUVMaps = bpy.context.window_manager.RizomUVBridgeUVMaps

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Tests if the operator is able to run.

        Args:
            context: Current data context.

        Returns:
            True if there is an active object.
        """

        return context.active_object is not None

    def get_objects(self) -> tuple[Object, list[Object], list[Object]]:
        """Build separate lists of scene objects.

        Returns:
            Active object, selected objects, selected objects with clones removed.
        """
        act_obj, sel_objs = obj_sel.get_selected()

        if self.prefs.exclude_clones:
            filtered_objs = [obj for obj in sel_objs if obj not in obj_sel.get_clones(sel_objs)]
        else:
            filtered_objs = sel_objs

        return act_obj, sel_objs, filtered_objs

    def repair_active_object(self, act_obj: Object) -> list[str]:
        """Scan the active object UV Maps for problems and fix them
        if detected.

        Args:
            act_obj: The active object.

        Returns:
            The names of the active object UV Maps.
        """
        for uvmap in act_obj.data.uv_layers:
            map_name: str = uvmap.name
            if "." in map_name:
                uvmap.name = map_name.replace(".", "_")
                self.act_obj_map_changes[map_name] = uvmap.name

        return [uvmap.name for uvmap in act_obj.data.uv_layers]

    def purge_and_repair_objects(self, control_names: list[str], scan_objs: list[Object]) -> None:
        """
        Scan objects and check that their UV maps match those in a validated list.
        If they do not purge all existing UV maps and remake them to match the validated list.
        This is destructive.

        Args:
            control_names: Validated list of UV map names.
            scan_objs: Objects to scan and purge.
        """
        for obj in scan_objs:
            if [uvmap.name for uvmap in obj.data.uv_layers] == control_names:
                continue

            self.changed_objs.append(obj.name)
            while obj.data.uv_layers:
                obj.data.uv_layers.remove(obj.data.uv_layers[0])

            for name in control_names:
                obj.data.uv_layers.new(name=name)

    def execute(self, context: Context) -> set[str]:
        """Execute the operator.

        Args:
            context: Current data context.

        Returns:
            Enum in {'RUNNING_MODAL', 'CANCELLED', 'FINISHED', 'FINISHED', 'INTERFACE'}
        """

        act_obj, _, chosen_objs = self.get_objects()

        act_obj_map_names = self.repair_active_object(act_obj)
        self.purge_and_repair_objects(act_obj_map_names, chosen_objs)

        if self.act_obj_map_changes:
            print("─" * 64)
            print(f"{act_obj.name} UV Map changes are documented below:")
            for original_name, new_name in self.act_obj_map_changes.items():
                print(f"{original_name} --> {new_name}")
            print("\n")

        if self.changed_objs:
            print(
                f"The following objects had their UV maps purged and recreated to match the UV maps of {act_obj.name}"
            )
            for obj in self.changed_objs:
                print(obj)
            print("─" * 64)
            print("\n")

        if self.act_obj_map_changes or self.changed_objs:
            self.report({"INFO"}, "UV Maps were made compatible with RizomUV, check the console for more information.")
            return {"FINISHED"}

        self.report({"INFO"}, "UV map names are compatible, no action needed.")

        return {"FINISHED"}
