"""Addon preferences."""

import platform
from pathlib import Path

import bpy
from bpy.props import BoolProperty, EnumProperty, FloatProperty, IntProperty, StringProperty
from bpy.types import Context

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.utility import json_utility, resolve_ruv_path

json_parser = json_utility.JsonUtility(constants.GUI_STRINGS)


# region Callbacks
def map_res_get(self) -> float:
    """Getter for the map_res property."""
    return self.get("map_res", 2048)


def map_res_set(self, value: int) -> None:
    """
    Setter for the map res property.
    Also updates the value of the margin and padding properties if map resolution changes.
    """
    old_value = self.get("map_res", 2048)
    self.margin = self.margin * (1 / old_value) * value
    self.padding = self.padding * (1 / old_value) * value

    self["map_res"] = value


# noinspection PyUnusedLocal
def update_addon_tab(self, context: Context) -> None:
    """Update the bl_category field of the addon UI."""

    if not self.category_name:
        self.category_name = "RizomUV"

    panel_ids = (
        "PANEL_PT_RizomUVBridge",
        "PANEL_PT_RizomUVBridgeRizomSettings",
        "PANEL_PT_RizomUVBridgeImportSettings",
        "PANEL_PT_RizomUVBridgeExportSettings",
        "PANEL_PT_RizomUVBridgeCoreSettings",
        "PANEL_PT_RizomUVBridgeAdvancedSettings",
        "PANEL_PT_RizomUVBridgeRemoteActions"
    )

    for p_id in panel_ids:
        cls = getattr(bpy.types, p_id)
        if cls.is_registered:
            bpy.utils.unregister_class(cls)
            cls.bl_category = self.category_name
            bpy.utils.register_class(cls)


# endregion


class RizomUVBridgePreferences(bpy.types.AddonPreferences):
    """RizomUV Bridge addon preferences."""

    bl_idname = constants.PACKAGE_NAME

    # region Addon settings.

    @staticmethod
    def default_rizom_path() -> str:
        """Calculate the default Rizom path."""

        rizom_path = "Installation not found, please manually select your rizomuv.exe installation."

        if platform.system() == "Windows":
            rizom_path = Path(resolve_ruv_path.win_get_latest_ruv()) / "rizomuv.exe"
            if rizom_path.is_file():
                return str(rizom_path)

        return rizom_path

    rizom_path: StringProperty(
        name="Rizom Executable",
        subtype="FILE_PATH",
        default=default_rizom_path(),
        description=json_parser.read_string("prefs", "rizom_path"),
    )

    confirm_export: BoolProperty(name="Confirm Export", description=json_parser.read_string("prefs", "confirm_export"))

    category_name: StringProperty(
        name="Category Name",
        default="RizomUV",
        description=json_parser.read_string("prefs", "category_name"),
        update=update_addon_tab,
        maxlen=32,
    )
    # endregion

    remote_actions: EnumProperty(
        name="Remote Actions",
        description=json_parser.read_string("prefs", "remote_actions"),
        items=(
            ("RESET_UVS", "Reset UVs", json_parser.read_string("actions", "RESET_UVS")),
            (
                "AUTOSEAMS_SHARP_EDGES",
                "Autoseams - Sharp Edges",
                json_parser.read_string("actions", "AUTOSEAMS_SHARP_EDGES"),
            ),
            ("AUTOSEAMS_MOSAIC", "Autoseams - Mosaic", json_parser.read_string("actions", "AUTOSEAMS_MOSAIC")),
            ("AUTOSEAMS_PELT", "Autoseams - Pelt", json_parser.read_string("actions", "AUTOSEAMS_PELT")),
            ("AUTOSEAMS_BOX", "Autoseams - Box", json_parser.read_string("actions", "AUTOSEAMS_BOX")),
            ("OFFSET_GROUPS", "Offset Groups", json_parser.read_string("actions", "OFFSET_GROUPS")),
        ),
    )

    # region Export settings.
    export_actions: EnumProperty(
        name="Export Actions",
        description=json_parser.read_string("prefs", "export_actions"),
        items=(
            ("EXPORT", "Export UVs", json_parser.read_string("actions", "EXPORT")),
            ("RESET_UVS", "Reset UVs", json_parser.read_string("actions", "RESET_UVS")),
            (
                "AUTOSEAMS_SHARP_EDGES",
                "Autoseams - Sharp Edges",
                json_parser.read_string("actions", "AUTOSEAMS_SHARP_EDGES"),
            ),
            ("AUTOSEAMS_MOSAIC", "Autoseams - Mosaic", json_parser.read_string("actions", "AUTOSEAMS_MOSAIC")),
            ("AUTOSEAMS_PELT", "Autoseams - Pelt", json_parser.read_string("actions", "AUTOSEAMS_PELT")),
            ("AUTOSEAMS_BOX", "Autoseams - Box", json_parser.read_string("actions", "AUTOSEAMS_BOX")),
        ),
    )

    apply_to_all_uvmaps_export: BoolProperty(
        name="Apply To All UVMaps",
        description=json_parser.read_string("action_settings", "apply_to_all_uvmaps_export"),
    )
    apply_to_all_uvmaps_action: BoolProperty(
        name="Apply To All UVMaps",
        description=json_parser.read_string("action_settings", "apply_to_all_uvmaps_action"),
    )

    exclude_clones: BoolProperty(
        name="Exclude Clones", default=True, description=json_parser.read_string("prefs", "exclude_clones")
    )

    file_type: EnumProperty(
        name="File Type",
        items=(
            ("FBX", "FBX File", json_parser.read_string("file_formats", ".fbx")),
            ("OBJ", "OBJ File", json_parser.read_string("file_formats", ".obj")),
        ),
        description=json_parser.read_string("prefs", "file_type"),
        default="FBX",
    )

    apply_settings_export: BoolProperty(
        "Apply Rizom Settings", description=json_parser.read_string("prefs", "apply_settings_export")
    )

    apply_settings_action: BoolProperty(
        "Apply Rizom Settings", description=json_parser.read_string("prefs", "apply_settings_action")
    )
    # endregion

    # region Import settings.
    mark_seams: BoolProperty(name="Mark Seams", description=json_parser.read_string("prefs", "mark_seams"))
    mark_sharp: BoolProperty(name="Mark Sharp", description=json_parser.read_string("prefs", "mark_sharp"))

    replace_objects: BoolProperty(
        name="Replace Objects", description=json_parser.read_string("prefs", "replace_objects")
    )

    reveal_hidden: BoolProperty(
        name="Reveal Hidden", default=True, description=json_parser.read_string("prefs", "reveal_hidden")
    )

    delete_uvmaps: BoolProperty(name="Delete UV Maps", description=json_parser.read_string("prefs", "delete_uvmaps"))

    triangulate: BoolProperty(name="Triangulate", description=json_parser.read_string("prefs", "triangulate"))

    fix_names: BoolProperty(name="Fix Names", description=json_parser.read_string("prefs", "fix_names"))
    # endregion

    # region Rizom settings.
    margin: FloatProperty(
        name="Margin",
        default=8.0,
        subtype="PIXEL",
        soft_max=32.0,
        min=0.0,
        step=1.0,
        description=json_parser.read_string("rizom_prefs", "margin"),
    )

    padding: FloatProperty(
        name="Padding",
        default=16.0,
        subtype="PIXEL",
        soft_max=32.0,
        min=0.0,
        step=1.0,
        description=json_parser.read_string("rizom_prefs", "padding"),
    )

    map_res: IntProperty(
        name="Map Resolution",
        min=1,
        default=2048,
        subtype="PIXEL",
        get=map_res_get,
        set=map_res_set,
        description=json_parser.read_string("rizom_prefs", "map_res"),
    )
    td_unit: EnumProperty(
        name="Texel Density Unit",
        description=json_parser.read_string("rizom_prefs", "td_unit"),
        items=(
            ("km", "tx/km", ""),
            ("m", "tx/m", ""),
            ("dm", "tx/dm", ""),
            ("cm", "tx/cm", ""),
            ("mm", "tx/mm", ""),
            ("in", "tx/in", ""),
            ("ft", "tx/ft", ""),
            ("yd", "tx/yd", ""),
            ("mi", "tx/mi", ""),
        ),
        default="cm",
    )
    td_target: FloatProperty(
        name="Texel Density Target",
        description=json_parser.read_string("rizom_prefs", "td_target"),
        default=10.24,
        min=0.0,
        subtype="PIXEL",
    )

    prevent_flips: BoolProperty(
        name="Prevent Flips", default=True, description=json_parser.read_string("rizom_prefs", "prevent_flips")
    )

    prevent_overlaps: BoolProperty(
        name="Prevent Overlaps", default=True, description=json_parser.read_string("rizom_prefs", "prevent_overlaps")
    )

    accuracy: IntProperty(
        name="Packing Accuracy",
        default=256,
        max=2048,
        min=128,
        description=json_parser.read_string("rizom_prefs", "accuracy"),
    )
    iterations: IntProperty(
        name="Iterations", default=1, min=1, max=128, description=json_parser.read_string("rizom_prefs", "iterations")
    )

    orientation: EnumProperty(
        name="Orientation",
        description=json_parser.read_string("rizom_prefs", "orientation"),
        items=(
            ("0", "No pre-orientation", "Don't pre-orient"),
            ("1", "Horizontal pre-orientation", "Pre-orient islands horizontally"),
            ("2", "Vertical pre-orientation", "Pre-orient islands vertically"),
            ("3", "X Axis pre-orientation", "Pre-orient islands so 3D X axis aligns with UV V axis"),
            ("4", "Y Axis pre-orientation", "Pre-orient islands so 3D Y axis aligns with UV V axis"),
            ("5", "Y Axis pre-orientation", "Pre-orient islands so 3D Z axis aligns with UV V axis"),
        ),
        default="0",
    )
    step_angle: IntProperty(
        name="Step Angle",
        description=json_parser.read_string("rizom_prefs", "step_angle"),
        subtype="ANGLE",
        min=0,
        max=180,
    )
    # endregion

    # region Actions settings.

    apply_settings_action: BoolProperty(
        "Apply Rizom Settings", description=json_parser.read_string("prefs", "apply_settings_action")
    )

    weld_before_autoseams: BoolProperty(
        name="Weld Before Autoseams",
        default=True,
        description=json_parser.read_string("action_settings", "weld_before_autoseams"),
    )

    stretch_quality: FloatProperty(
        name="Stretch Limiter",
        default=0.25,
        min=0,
        max=0.99,
        step=0.1,
        description=json_parser.read_string("action_settings", "stretch_quality"),
    )

    link_gaps: BoolProperty(name="Link Gaps", description=json_parser.read_string("action_settings", "link_gaps"))
    open_cylinders: BoolProperty(
        name="Open Cylinders", description=json_parser.read_string("action_settings", "open_cylinders")
    )
    cut_handles: BoolProperty(name="Cut Handles", description=json_parser.read_string("action_settings", "cut_handles"))

    sharp_angle: IntProperty(
        name="Edge Angle",
        default=70,
        min=0,
        max=180,
        description=json_parser.read_string("action_settings", "sharp_angle"),
    )

    mosaic_force: FloatProperty(
        name="Force",
        default=0.5,
        step=0.05,
        min=0.01,
        max=0.99,
        description=json_parser.read_string("action_settings", "mosaic_force"),
    )

    pelt_trunk: BoolProperty(name="Tree", description=json_parser.read_string("action_settings", "pelt_trunk"))
    pelt_branch: BoolProperty(name="Branch", description=json_parser.read_string("action_settings", "pelt_branch"))
    pelt_leaf: BoolProperty(name="Leaf", description=json_parser.read_string("action_settings", "pelt_leaf"))

    group_offset_value: IntProperty(
        name="Offset", default=1, description=json_parser.read_string("action_settings", "group_offset_value")
    )
    # endregion
