"""These property groups are not saved to the blender preferences file."""

import bpy
from bpy.props import EnumProperty
from bpy.types import Context

from rizomuv_bridge_link.data import constants
from rizomuv_bridge_link.functions.utility import json_utility

json_parser = json_utility.JsonUtility(constants.GUI_STRINGS)


class RizomUVBridgeGUITabs(bpy.types.PropertyGroup):
    """Supplies the different UI tabs available to the user."""

    tabs: EnumProperty(
        name="Panel Tabs",
        description=json_parser.read_string("props", "panel_tabs"),
        items=(
            ("OPERATIONS", "Operations Tab", "", "EXPORT", 0),
            ("RIZOM_LIVE", "Rizom Live Remote Tab", "", "TEXT", 1),
            ("ADDON_SETTINGS", "Addon Settings Tab", "", "PREFERENCES", 2),
        ),
    )


# <editor-fold desc="Dynamic UV Map property.">
# noinspection PyUnusedLocal
def get_uvmaps(self, context: Context) -> list[tuple[str, str, str]]:
    """
    Build a list of UV Maps from the active object to display in the UI.
    Note: self/context parameters are required.

    Args:
        self: RizomUVBridgeUVMaps.
        context: Current data context.

    Returns:
        List of tuples containing data for the EnumProperty to display. (Identifier, Name, Description)

    """
    act_obj = context.active_object

    if not act_obj or act_obj.type != "MESH":
        return [("NO_OBJ", "INFO: No active object", "")]

    act_obj_uv_layers = act_obj.data.uv_layers
    if act_obj_uv_layers:
        items = []
        for uv_map in act_obj_uv_layers:
            name = uv_map.name
            items.append((name, name, ""))

        # This is required due to a bug with the blender API ->
        # https://docs.blender.org/api/master/bpy.props.html#bpy.props.EnumProperty
        RizomUVBridgeUVMaps.uv_maps_list = items
        return items

    return [("NO_MAP", "INFO: The active object has no UV Maps", "")]


# noinspection PyUnusedLocal
def uvmap_update(self, context: Context) -> None:
    """
    Function is run if the value of RizomUVBridgeUVMaps.uv_maps is changed.
    Syncs the active UV map of the active object with the one shown on the addon UI.

    Args:
        self: RizomUVBridgeUVMaps.
        context: Current data context.

    """

    maps = context.window_manager.RizomUVBridgeUVMaps
    act_obj = context.active_object

    if act_obj and act_obj.type == "MESH":
        act_obj_uv_layers = act_obj.data.uv_layers
        act_obj_uv_layers.active = act_obj_uv_layers[maps.uv_maps]


# noinspection PyTypeChecker
class RizomUVBridgeUVMaps(bpy.types.PropertyGroup):
    """Display and set the active UV map from the addon."""

    uv_maps_list = []
    uv_maps: EnumProperty(
        name="UV Map",
        description=json_parser.read_string("props", "uv_maps"),
        update=uvmap_update,
        items=get_uvmaps,
    )


# </editor-fold>
